/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.favorites.history;

import com.google.common.base.Suppliers;
import me.shedaniel.clothconfig2.api.LazyResettable;
import me.shedaniel.clothconfig2.api.animator.ValueAnimator;
import me.shedaniel.math.Dimension;
import me.shedaniel.math.FloatingRectangle;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.widgets.*;
import me.shedaniel.rei.api.client.registry.category.CategoryRegistry;
import me.shedaniel.rei.api.client.registry.display.DisplayCategory;
import me.shedaniel.rei.api.client.util.MatrixUtils;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.impl.client.ClientHelperImpl;
import me.shedaniel.rei.impl.client.gui.widget.AutoCraftingEvaluator;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_364;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector3f;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

public class DisplayEntry extends WidgetWithBounds {
    private final LazyResettable<List<Widget>> widgets = new LazyResettable<>(this::setupWidgets);
    private final DisplayHistoryWidget parent;
    private final Display display;
    private final Dimension size = new Dimension(1, 1);
    private final Supplier<AutoCraftingEvaluator.AutoCraftingResult> autoCraftingResult =
            Suppliers.memoizeWithExpiration(this::evaluateAutoCrafting, 1000, TimeUnit.MILLISECONDS);
    private boolean hasInitialBounds;
    private final ValueAnimator<FloatingRectangle> bounds = ValueAnimator.ofFloatingRectangle();
    private final Button plusButton;
    private double xOffset = 0;
    private boolean reachedStable = false;
    private UUID uuid = UUID.randomUUID();
    
    public DisplayEntry(DisplayHistoryWidget parent, Display display, @Nullable Rectangle initialBounds) {
        this.display = display;
        this.parent = parent;
        this.hasInitialBounds = initialBounds != null;
        if (this.hasInitialBounds) {
            this.bounds.setAs(initialBounds.getFloatingBounds());
            this.plusButton = Widgets.createButton(new Rectangle(initialBounds.getMaxX() - 16, initialBounds.getMaxY() - 16, 10, 10), class_2561.method_43470("+"));
        } else {
            this.plusButton = Widgets.createButton(new Rectangle(-1000, -1000, 10, 10), class_2561.method_43470("+"));
        }
    }
    
    private AutoCraftingEvaluator.AutoCraftingResult evaluateAutoCrafting() {
        if (this.display == null) return new AutoCraftingEvaluator.AutoCraftingResult();
        return AutoCraftingEvaluator.evaluateAutoCrafting(false, false, this.display, this.display::provideInternalDisplayIds);
    }
    
    public UUID getUuid() {
        return uuid;
    }
    
    public void setUuid(UUID uuid) {
        this.uuid = uuid;
    }
    
    public void markBoundsDirty() {
        widgets.reset();
    }
    
    private List<Widget> setupWidgets() {
        Rectangle parentBounds = parent.getBounds();
        CategoryRegistry.CategoryConfiguration<Display> configuration = CategoryRegistry.getInstance().get((CategoryIdentifier<Display>) display.getCategoryIdentifier());
        DisplayCategory<Display> category = configuration.getCategory();
        Rectangle displayBounds = new Rectangle(0, 0, category.getDisplayWidth(display), category.getDisplayHeight());
        List<Widget> widgets = setupDisplay(configuration, displayBounds);
        float scale = 1.0F;
        if (parentBounds.width * scale < displayBounds.width) {
            scale = Math.min(scale, parentBounds.width * scale / (float) displayBounds.width);
        }
        if (parentBounds.height * scale < displayBounds.height) {
            scale = Math.min(scale, parentBounds.height * scale / (float) displayBounds.height);
        }
        float x = parentBounds.getCenterX() - displayBounds.width / 2 * scale;
        float y = parentBounds.getCenterY() - displayBounds.height / 2 * scale;
        FloatingRectangle newBounds = new Rectangle(x, y, displayBounds.width * scale, displayBounds.height * scale).getFloatingBounds();
        if (hasInitialBounds) {
            if (this.size.width == 1 && this.size.height == 1 && !ConfigObject.getInstance().isReducedMotion()) {
                this.bounds.setTo(newBounds, 700);
            } else {
                this.bounds.setAs(newBounds);
            }
        } else {
            this.bounds.setAs(newBounds);
            hasInitialBounds = true;
        }
        this.size.setSize(displayBounds.getSize());
        return widgets;
    }
    
    private List<Widget> setupDisplay(CategoryRegistry.CategoryConfiguration<Display> configuration, Rectangle displayBounds) {
        try {
            return configuration.getView(display).setupDisplay(display, displayBounds);
        } catch (Throwable throwable) {
            throwable.printStackTrace();
            List<Widget> w = new ArrayList<>();
            w.add(Widgets.createRecipeBase(displayBounds).color(0xFFBB0000));
            w.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() - 8), class_2561.method_43470("Failed to initiate setupDisplay")));
            w.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() + 1), class_2561.method_43470("Check console for error")));
            return w;
        }
    }
    
    @Override
    public Rectangle getBounds() {
        return bounds.value().getBounds();
    }
    
    public Dimension getSize() {
        return size;
    }
    
    public boolean isStable() {
        widgets.get();
        FloatingRectangle target = this.bounds.target();
        FloatingRectangle value = this.bounds.value();
        return reachedStable || Math.abs(value.x - target.x) <= 0.5 && Math.abs(value.y - target.y) <= 0.5 && Math.abs(value.width - target.width) <= 0.5 && Math.abs(value.height - target.height) <= 0.5;
    }
    
    public void setReachedStable(boolean reachedStable) {
        this.reachedStable = reachedStable;
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        boolean stable = isStable();
        this.bounds.update(delta);
        FloatingRectangle target = this.bounds.target();
        FloatingRectangle bounds = this.bounds.value();
        
        if (!reachedStable && Math.abs(bounds.x - target.x) <= 0.5 && Math.abs(bounds.y - target.y) <= 0.5 && Math.abs(bounds.width - target.width) <= 0.5 && Math.abs(bounds.height - target.height) <= 0.5) {
            reachedStable = true;
        }
        
        if (stable && (bounds.getMaxX() + xOffset < parent.getBounds().x || bounds.x + xOffset > parent.getBounds().getMaxX())) {
            return;
        }
        
        graphics.method_51448().pushMatrix();
        graphics.method_51448().translate(xOffset(), yOffset());
        graphics.method_51448().scale(xScale(), yScale());
        
        for (Widget widget : widgets.get()) {
            widget.method_25394(graphics, transformMouseX(mouseX), transformMouseY(mouseY), delta);
        }
        graphics.method_51448().popMatrix();
        
        {
            graphics.method_51448().pushMatrix();
            if (stable && target.equals(bounds)) {
                graphics.method_51448().translate((float) this.xOffset, 0);
            }
            Vector3f mouse = new Vector3f((float) mouseX, (float) mouseY, 1);
            graphics.method_51448().transform(mouse);
            
            AutoCraftingEvaluator.AutoCraftingResult result = this.autoCraftingResult.get();
            
            plusButton.setEnabled(result.successful);
            plusButton.setTint(result.tint);
            plusButton.getBounds().setBounds(new Rectangle(bounds.getMaxX() - 14, bounds.getMaxY() - 14, 10, 10));
            
            if (result.hasApplicable) {
                plusButton.setText(class_2561.method_43470("+"));
                plusButton.method_25394(graphics, Math.round(mouse.x()), Math.round(mouse.y()), delta);
                graphics.method_51448().popMatrix();
                
                if (plusButton.containsMouse(Math.round(mouse.x()), Math.round(mouse.y()))) {
                    result.tooltipRenderer.accept(new Point(mouseX, mouseY), Tooltip::queue);
                }
                
                if (result.renderer != null) {
                    graphics.method_51448().pushMatrix();
                    graphics.method_51448().translate(xOffset(), yOffset());
                    graphics.method_51448().scale(xScale(), yScale());
                    
                    Rectangle transformedBounds = MatrixUtils.transform(MatrixUtils.inverse(graphics.method_51448()), getBounds());
                    result.renderer.render(graphics, mouseX, mouseY, delta, widgets.get(), transformedBounds, display);
                    graphics.method_51448().popMatrix();
                }
            } else {
                graphics.method_51448().popMatrix();
            }
        }
    }
    
    @Override
    public boolean method_25402(class_11909 event, boolean doubleClick) {
        if (containsMouse(event.comp_4798() + xOffset, event.comp_4799())) {
            for (Widget widget : widgets.get()) {
                if (widget.method_25402(new class_11909(transformMouseX(event.comp_4798()), transformMouseY(event.comp_4799()), event.comp_4800()), doubleClick)) {
                    return true;
                }
            }
            
            return true;
        }
        
        return false;
    }
    
    @Override
    public boolean method_25406(class_11909 event) {
        if (containsMouse(event.comp_4798() + xOffset, event.comp_4799())) {
            for (Widget widget : widgets.get()) {
                if (widget.method_25406(new class_11909(transformMouseX(event.comp_4798()), transformMouseY(event.comp_4799()), event.comp_4800()))) {
                    return true;
                }
            }
            
            if (event.method_74245() == 0 && plusButton.containsMouse(event.comp_4798() + xOffset, event.comp_4799())) {
                AutoCraftingEvaluator.evaluateAutoCrafting(true, event.method_74239(), display, display::provideInternalDisplayIds);
                Widgets.produceClickSound();
                return true;
            }
            
            ClientHelperImpl.getInstance()
                    .openDisplayViewingScreen(Map.of(CategoryRegistry.getInstance().get(display.getCategoryIdentifier()).getCategory(), List.of(display)),
                            null, List.of(), List.of());
            Widgets.produceClickSound();
            return true;
        }
        
        return super.method_25406(event);
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        try {
            Widget.pushMouse(new Point(transformMouseX(mouse().x), transformMouseY(mouse().y)));
            for (Widget widget : widgets.get()) {
                if (widget.method_25404(event)) {
                    return true;
                }
            }
        } finally {
            Widget.popMouse();
        }
        
        return super.method_25404(event);
    }
    
    private float xOffset() {
        FloatingRectangle bounds = this.bounds.value();
        FloatingRectangle target = this.bounds.target();
        float xOffset = (float) bounds.x;
        if (isStable() && target.equals(bounds)) {
            xOffset += this.xOffset;
        }
        return xOffset;
    }
    
    private float yOffset() {
        FloatingRectangle bounds = this.bounds.value();
        return (float) bounds.y;
    }
    
    private float xScale() {
        FloatingRectangle bounds = this.bounds.value();
        return (float) bounds.width / size.width;
    }
    
    private float yScale() {
        FloatingRectangle bounds = this.bounds.value();
        return (float) bounds.height / size.height;
    }
    
    protected int transformMouseX(int mouseX) {
        return Math.round((mouseX - xOffset()) / xScale());
    }
    
    protected int transformMouseY(int mouseY) {
        return Math.round((mouseY - yOffset()) / yScale());
    }
    
    protected double transformMouseX(double mouseX) {
        return (mouseX - xOffset()) / xScale();
    }
    
    protected double transformMouseY(double mouseY) {
        return (mouseY - yOffset()) / yScale();
    }
    
    @Override
    public List<? extends class_364> method_25396() {
        return Collections.emptyList();
    }
    
    public void setScrolled(double xOffset) {
        this.xOffset = xOffset;
    }
    
    public List<Widget> getWidgets() {
        return widgets.get();
    }
    
    public Display getDisplay() {
        return display;
    }
}
