/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.basewidgets;

import me.shedaniel.clothconfig2.api.TickableWidget;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.gui.widgets.TextField;
import me.shedaniel.rei.api.client.gui.widgets.Widget;
import me.shedaniel.rei.api.client.gui.widgets.WidgetWithBounds;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_156;
import net.minecraft.class_2583;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_3544;
import net.minecraft.class_5481;
import net.minecraft.class_8016;
import net.minecraft.class_8023;
import net.minecraft.class_9848;
import net.minecraft.util.*;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

@ApiStatus.Internal
public class TextFieldWidget extends WidgetWithBounds implements TickableWidget, TextField {
    public Function<String, String> stripInvalid = class_3544::method_57180;
    protected int frame;
    protected boolean editable = true;
    protected int firstCharacterIndex;
    protected int cursorPos;
    protected int highlightPos;
    protected int editableColor = 0xffe0e0e0;
    protected int notEditableColor = 0xff707070;
    protected TextFormatter formatter = TextFormatter.DEFAULT;
    private Rectangle bounds;
    private String text = "";
    private int maxLength = 32;
    private boolean hasBorder = true;
    private boolean focusUnlocked = true;
    private boolean focused = false;
    private boolean visible = true;
    private boolean selecting = false;
    @Nullable
    private String suggestion;
    @Nullable
    private Consumer<String> responder;
    private Predicate<String> filter = s -> true;
    
    public TextFieldWidget(Rectangle bounds) {
        this.bounds = bounds;
    }
    
    public TextFieldWidget(int x, int y, int width, int height) {
        this(new Rectangle(x, y, width, height));
    }
    
    public String getSuggestion() {
        return suggestion;
    }
    
    public void setSuggestion(String suggestion) {
        this.suggestion = suggestion;
    }
    
    @Override
    public Rectangle getBounds() {
        return bounds;
    }
    
    public void setResponder(Consumer<String> responder) {
        this.responder = responder;
    }
    
    public void setFormatter(TextFormatter formatter) {
        this.formatter = formatter;
    }
    
    @Override
    public void tick() {
        this.frame++;
    }
    
    @Override
    public String getText() {
        return this.text;
    }
    
    @Override
    public void setText(String text) {
        if (this.filter.test(text)) {
            if (text.length() > this.maxLength) {
                this.text = text.substring(0, this.maxLength);
            } else {
                this.text = text;
            }
            
            this.onChanged(text);
            this.moveCursorToEnd();
        }
    }
    
    @Override
    public String getSelectedText() {
        int i = Math.min(this.cursorPos, this.highlightPos);
        int j = Math.max(this.cursorPos, this.highlightPos);
        return this.text.substring(i, j);
    }
    
    public void setFilter(Predicate<String> filter) {
        this.filter = filter;
    }
    
    @Override
    public void addText(String text) {
        int highlightStart = Math.min(this.cursorPos, this.highlightPos);
        int highlightEnd = Math.max(this.cursorPos, this.highlightPos);
        int k = this.maxLength - this.text.length() - (highlightStart - highlightEnd);
        String textFiltered = stripInvalid.apply(text);
        int l = textFiltered.length();
        if (k < l) {
            textFiltered = textFiltered.substring(0, k);
            l = k;
        }
        
        String result = new StringBuilder(this.text)
                .replace(highlightStart, highlightEnd, textFiltered)
                .toString();
        if (this.filter.test(result)) {
            this.text = result;
            this.setCursorPosition(highlightStart + l);
            this.setHighlightPos(this.cursorPos);
            this.onChanged(this.text);
        }
    }
    
    public void onChanged(String newText) {
        if (this.responder != null) {
            this.responder.accept(newText);
        }
    }
    
    private void erase(int offset) {
        if (class_310.method_1551().method_74188()) {
            this.eraseWords(offset);
        } else {
            this.eraseCharacters(offset);
        }
    }
    
    public void eraseWords(int wordOffset) {
        if (!this.text.isEmpty()) {
            if (this.highlightPos != this.cursorPos) {
                this.addText("");
            } else {
                this.eraseCharacters(this.getWordPosition(wordOffset) - this.cursorPos);
            }
        }
    }
    
    public void eraseCharacters(int characterOffset) {
        if (!this.text.isEmpty()) {
            if (this.highlightPos != this.cursorPos) {
                this.addText("");
            } else {
                int offsetCursorPos = this.getCursorPos(characterOffset);
                int from = Math.min(offsetCursorPos, this.cursorPos);
                int to = Math.max(offsetCursorPos, this.cursorPos);
                if (from != to) {
                    String string = new StringBuilder(this.text).delete(from, to).toString();
                    if (this.filter.test(string)) {
                        this.text = string;
                        this.moveCursorTo(from);
                    }
                }
                this.onChanged(this.text);
            }
        }
    }
    
    public int getWordPosition(int wordOffset) {
        return this.getWordPosition(wordOffset, this.getCursor());
    }
    
    public int getWordPosition(int wordOffset, int cursorPosition) {
        return this.getWordPosition(wordOffset, cursorPosition, true);
    }
    
    public int getWordPosition(int wordOffset, int cursorPosition, boolean skipOverSpaces) {
        int cursor = cursorPosition;
        boolean backwards = wordOffset < 0;
        int absoluteOffset = Math.abs(wordOffset);
        
        for (int k = 0; k < absoluteOffset; ++k) {
            if (!backwards) {
                int l = this.text.length();
                cursor = this.text.indexOf(32, cursor);
                if (cursor == -1) {
                    cursor = l;
                } else {
                    while (skipOverSpaces && cursor < l && this.text.charAt(cursor) == ' ') {
                        ++cursor;
                    }
                }
            } else {
                while (skipOverSpaces && cursor > 0 && this.text.charAt(cursor - 1) == ' ') {
                    --cursor;
                }
                
                while (cursor > 0 && this.text.charAt(cursor - 1) != ' ') {
                    --cursor;
                }
            }
        }
        
        return cursor;
    }
    
    public void moveCursor(int by) {
        this.moveCursorTo(this.cursorPos + by);
    }
    
    private int getCursorPos(int i) {
        return class_156.method_27761(this.text, this.cursorPos, i);
    }
    
    @Override
    public void moveCursorTo(int cursor) {
        this.setCursorPosition(cursor);
        if (!selecting) {
            this.setHighlightPos(this.cursorPos);
        }
    }
    
    @Override
    public void moveCursorToStart() {
        this.moveCursorTo(0);
    }
    
    @Override
    public void moveCursorToEnd() {
        this.moveCursorTo(this.text.length());
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        if (this.isVisible() && this.method_25370()) {
            this.selecting = event.method_74239();
            if (event.method_74241()) {
                this.moveCursorToEnd();
                this.setHighlightPos(0);
                return true;
            } else if (event.method_74242()) {
                minecraft.field_1774.method_1455(this.getSelectedText());
                return true;
            } else if (event.method_74243()) {
                if (this.editable) {
                    this.addText(minecraft.field_1774.method_1460());
                }
                
                return true;
            } else if (event.method_74244()) {
                minecraft.field_1774.method_1455(this.getSelectedText());
                if (this.editable) {
                    this.addText("");
                }
                
                return true;
            } else {
                switch (event.comp_4795()) {
                    case 259:
                        if (this.editable) {
                            this.selecting = false;
                            this.erase(-1);
                            this.selecting = event.method_74239();
                        }
                        
                        return true;
                    case 260:
                    case 264:
                    case 265:
                    case 266:
                    case 267:
                    default:
                        return event.comp_4795() != 256;
                    case 261:
                        if (this.editable) {
                            this.selecting = false;
                            this.erase(1);
                            this.selecting = event.method_74239();
                        }
                        
                        return true;
                    case 262:
                        if (event.method_74240()) {
                            this.moveCursorTo(this.getWordPosition(1));
                        } else {
                            this.moveCursor(1);
                        }
                        
                        return true;
                    case 263:
                        if (event.method_74240()) {
                            this.moveCursorTo(this.getWordPosition(-1));
                        } else {
                            this.moveCursor(-1);
                        }
                        
                        return true;
                    case 268:
                        this.moveCursorToStart();
                        return true;
                    case 269:
                        this.moveCursorToEnd();
                        return true;
                }
            }
        } else {
            return false;
        }
    }
    
    @Override
    public boolean method_25400(class_11905 event) {
        if (this.isVisible() && this.method_25370()) {
            if (event.method_74227() && !(
                    class_310.method_1551().method_74188() && !class_310.method_1551().method_74187() && !class_310.method_1551().method_74189() && (
                            event.comp_4793() == 'a' || event.comp_4793() == 'c' || event.comp_4793() == 'v'
                    )
            )) {
                if (this.editable) {
                    this.addText(event.method_74226());
                }
                
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }
    
    @Override
    public List<Widget> method_25396() {
        return Collections.emptyList();
    }
    
    @Override
    public boolean method_25402(class_11909 event, boolean doubleClick) {
        if (!this.isVisible()) {
            return false;
        } else {
            boolean hovered = event.comp_4798() >= (double) this.bounds.x && event.comp_4798() < (double) (this.bounds.x + this.bounds.width) && event.comp_4799() >= (double) this.bounds.y && event.comp_4799() < (double) (this.bounds.y + this.bounds.height);
            if (this.focusUnlocked) {
                this.method_25365(hovered);
            }
            
            if (this.focused && hovered && event.method_74245() == 0) {
                int int_2 = class_3532.method_15357(event.comp_4798()) - this.bounds.x;
                if (this.hasBorder) {
                    int_2 -= 4;
                }
                
                String string_1 = this.font.method_27523(this.text.substring(this.firstCharacterIndex), this.getWidth());
                this.moveCursorTo(this.font.method_27523(string_1, int_2).length() + this.firstCharacterIndex);
                return true;
            } else {
                return false;
            }
        }
    }
    
    public void renderBorder(class_332 graphics) {
        if (this.hasBorder()) {
            int borderColor = containsMouse(mouse()) || focused ? 0xffffffff : 0xffa0a0a0;
            graphics.method_25294(this.bounds.x - 1, this.bounds.y - 1, this.bounds.x + this.bounds.width + 1, this.bounds.y + this.bounds.height + 1, 0xff000000);
            graphics.method_25294(this.bounds.x, this.bounds.y, this.bounds.x + this.bounds.width, this.bounds.y + this.bounds.height, borderColor);
            graphics.method_25294(this.bounds.x + 1, this.bounds.y + 1, this.bounds.x + this.bounds.width - 1, this.bounds.y + this.bounds.height - 1, 0xff000000);
        }
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        if (this.isVisible()) {
            this.renderBorder(graphics);
            
            int color = this.editable ? this.editableColor : this.notEditableColor;
            int int_4 = this.cursorPos - this.firstCharacterIndex;
            int int_5 = this.highlightPos - this.firstCharacterIndex;
            String textClipped = this.font.method_27523(this.text.substring(this.firstCharacterIndex), this.getWidth());
            boolean boolean_1 = int_4 >= 0 && int_4 <= textClipped.length();
            boolean boolean_2 = this.focused && this.frame / 6 % 2 == 0 && boolean_1;
            int x = this.hasBorder ? this.bounds.x + 4 : this.bounds.x;
            int y = this.hasBorder ? this.bounds.y + (this.bounds.height - 8) / 2 : this.bounds.y;
            int int_8 = x;
            int_5 = Math.min(textClipped.length(), int_5);
            
            if (!textClipped.isEmpty()) {
                String string_2 = boolean_1 ? textClipped.substring(0, int_4) : textClipped;
                class_5481 sequence = this.formatter.format(this, string_2, this.firstCharacterIndex);
                graphics.method_35720(this.font, sequence, x, y, color);
                int_8 = x + this.font.method_30880(sequence);
            }
            
            boolean isCursorInsideText = this.cursorPos < this.text.length() || this.text.length() >= this.getMaxLength();
            int selectionLeft = int_8;
            if (!boolean_1) {
                selectionLeft = int_4 > 0 ? x + this.bounds.width : x;
            } else if (isCursorInsideText) {
                --int_8;
            }
            selectionLeft--;
            
            if (!textClipped.isEmpty() && boolean_1 && int_4 < textClipped.length()) {
                graphics.method_35720(this.font, this.formatter.format(this, textClipped.substring(int_4), this.cursorPos), int_8, y, color);
            }
            
            if (!isCursorInsideText && text.isEmpty() && this.suggestion != null) {
                renderSuggestion(graphics, x, y);
            }
            
            if (boolean_2) {
                graphics.method_25294(selectionLeft + 1, y, selectionLeft + 2, y + 9, ((0xFF) << 24) | ((((color >> 16 & 255) / 4) & 0xFF) << 16) | ((((color >> 8 & 255) / 4) & 0xFF) << 8) | ((((color & 255) / 4) & 0xFF)));
                graphics.method_25294(selectionLeft, y - 1, selectionLeft + 1, y + 8, ((0xFF) << 24) | color);
            }
            
            // Render selection overlay
            if (int_5 != int_4) {
                int selectionRight = x + this.font.method_1727(textClipped.substring(0, int_5));
                this.renderSelection(graphics, selectionLeft, y - 1, selectionRight - 1, y + 9, color);
            }
        }
    }
    
    protected void renderSuggestion(class_332 graphics, int x, int y) {
        graphics.method_25303(this.font, this.font.method_27523(this.suggestion, this.getWidth()), x, y, -8355712);
    }
    
    protected void renderSelection(class_332 graphics, int x1, int y1, int x2, int y2, int color) {
        int tmp;
        if (x1 < x2) {
            tmp = x1;
            x1 = x2;
            x2 = tmp;
        }
        
        if (y1 < y2) {
            tmp = y1;
            y1 = y2;
            y2 = tmp;
        }
        
        if (x2 > this.bounds.x + this.bounds.width) {
            x2 = this.bounds.x + this.bounds.width;
        }
        
        if (x1 > this.bounds.x + this.bounds.width) {
            x1 = this.bounds.x + this.bounds.width;
        }
        
        int finalX1 = x1, finalX2 = x2, finalY1 = y1, finalY2 = y2;
        graphics.method_25296(x1, y1, x2, y2, class_9848.method_61330(120, color), class_9848.method_61330(120, color));
    }
    
    @Override
    public int getMaxLength() {
        return this.maxLength;
    }
    
    @Override
    public void setMaxLength(int maxLength) {
        this.maxLength = maxLength;
        if (this.text.length() > maxLength) {
            this.text = this.text.substring(0, maxLength);
            this.onChanged(this.text);
        }
    }
    
    @Override
    public int getCursor() {
        return this.cursorPos;
    }
    
    @Override
    public void setCursorPosition(int cursor) {
        this.cursorPos = class_3532.method_15340(cursor, 0, this.text.length());
    }
    
    @Override
    public boolean hasBorder() {
        return this.hasBorder;
    }
    
    @Override
    public void setHasBorder(boolean hasBorder) {
        this.hasBorder = hasBorder;
    }
    
    @Override
    public void setEditableColor(int editableColor) {
        this.editableColor = editableColor;
    }
    
    @Override
    public void setNotEditableColor(int notEditableColor) {
        this.notEditableColor = notEditableColor;
    }
    
    @Nullable
    public class_8016 method_48205(class_8023 event) {
        return this.visible && this.editable ? super.method_48205(event) : null;
    }
    
    @Override
    public boolean method_25370() {
        return this.focused;
    }
    
    @Override
    public void method_25365(boolean focused) {
        if (focused && !this.focused)
            this.frame = 0;
        this.focused = focused;
    }
    
    public void setIsEditable(boolean isEditable) {
        this.editable = isEditable;
    }
    
    public int getWidth() {
        return this.hasBorder() ? this.bounds.width - 8 : this.bounds.width;
    }
    
    public void setHighlightPos(int highlightPos) {
        int j = this.text.length();
        this.highlightPos = class_3532.method_15340(highlightPos, 0, j);
        if (this.font != null) {
            if (this.firstCharacterIndex > j) {
                this.firstCharacterIndex = j;
            }
            
            int width = this.getWidth();
            String clippedText = this.font.method_27523(this.text.substring(this.firstCharacterIndex), width);
            int int_4 = clippedText.length() + this.firstCharacterIndex;
            if (this.highlightPos == this.firstCharacterIndex) {
                this.firstCharacterIndex -= this.font.method_27524(this.text, width, true).length();
            }
            
            if (this.highlightPos > int_4) {
                this.firstCharacterIndex += this.highlightPos - int_4;
            } else if (this.highlightPos <= this.firstCharacterIndex) {
                this.firstCharacterIndex -= this.firstCharacterIndex - this.highlightPos;
            }
            
            this.firstCharacterIndex = class_3532.method_15340(this.firstCharacterIndex, 0, j);
        }
        
    }
    
    public void setFocusUnlocked(boolean focusUnlocked) {
        this.focusUnlocked = focusUnlocked;
    }
    
    public boolean isVisible() {
        return this.visible;
    }
    
    public void setVisible(boolean visible) {
        this.visible = visible;
    }
    
    public int getCharacterX(int index) {
        return index > this.text.length() ? this.bounds.x : this.bounds.x + this.font.method_1727(this.text.substring(0, index));
    }
    
    public interface TextFormatter {
        TextFormatter DEFAULT = (widget, text, index) -> {
            return class_5481.method_30747(text, class_2583.field_24360);
        };
        
        class_5481 format(TextFieldWidget widget, String text, int index);
    }
}
