/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.common.networking;

import dev.architectury.networking.NetworkManager;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import me.shedaniel.rei.RoughlyEnoughItemsNetwork;
import me.shedaniel.rei.api.client.registry.display.DisplayRegistry;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.impl.client.registry.display.DisplayRegistryImpl;
import me.shedaniel.rei.impl.common.InternalLogger;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_7995;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;
import java.util.function.IntFunction;
import java.util.function.UnaryOperator;

public record DisplaySyncPacket(SyncType syncType, Collection<Display> displays, long version) implements class_8710 {
    public static final class_9154<DisplaySyncPacket> TYPE = new class_9154<>(RoughlyEnoughItemsNetwork.SYNC_DISPLAYS_PACKET);
    public static final class_9139<? super class_9129, DisplaySyncPacket> STREAM_CODEC = class_9139.method_56436(
            SyncType.STREAM_CODEC,
            DisplaySyncPacket::syncType,
            Display.streamCodec().method_56433(codec -> {
                return new class_9139<class_9129, Display>() {
                    @Override
                    public void encode(class_9129 buf, Display display) {
                        class_9129 tmpBuf = new class_9129(Unpooled.buffer(), buf.method_56349());
                        try {
                            codec.encode(tmpBuf, display);
                        } catch (Throwable throwable) {
                            tmpBuf.release();
                            buf.method_52964(false);
                            InternalLogger.getInstance().debug("Failed to encode display: %s".formatted(display), throwable);
                            return;
                        }
                        
                        buf.method_52964(true);
                        class_9129.method_56894(buf, ByteBufUtil.getBytes(tmpBuf));
                        tmpBuf.release();
                    }
                    
                    @Override
                    public Display decode(class_9129 buf) {
                        boolean success = buf.readBoolean();
                        if (!success) return null;
                        class_9129 tmpBuf = new class_9129(Unpooled.wrappedBuffer(class_9129.method_56890(buf)), buf.method_56349());
                        try {
                            return codec.decode(tmpBuf);
                        } catch (Throwable throwable) {
                            return null;
                        } finally {
                            tmpBuf.release();
                        }
                    }
                };
            }).method_56433(class_9135.<class_9129, Display, Collection<Display>>method_56374(ArrayList::new)).method_56432(list -> {
                return list.stream()
                        .filter(Objects::nonNull)
                        .toList();
            }, UnaryOperator.identity()),
            DisplaySyncPacket::displays,
            class_9135.field_54505,
            DisplaySyncPacket::version,
            DisplaySyncPacket::new
    );
    
    @Override
    public class_9154<? extends class_8710> method_56479() {
        return DisplaySyncPacket.TYPE;
    }
    
    @Environment(EnvType.CLIENT)
    public void handle(NetworkManager.PacketContext context) {
        DisplayRegistryImpl registry = (DisplayRegistryImpl) DisplayRegistry.getInstance();
        if (syncType() == SyncType.SET) {
            InternalLogger.getInstance().info("[REI Server Display Sync] Received server's request to set %d recipes.", displays().size());
            registry.addJob(() -> {
                registry.removeSyncedRecipes();
                for (Display display : displays()) {
                    registry.add(display, DisplayRegistryImpl.SYNCED);
                }
            });
        } else if (syncType() == SyncType.APPEND) {
            InternalLogger.getInstance().info("[REI Server Display Sync] Received server's request to append %d recipes.", displays().size());
            registry.addJob(() -> {
                for (Display display : displays()) {
                    registry.add(display, DisplayRegistryImpl.SYNCED);
                }
            });
        }
    }
    
    public enum SyncType {
        APPEND,
        SET;
        
        public static final IntFunction<SyncType> BY_ID = class_7995.method_47914(Enum::ordinal, values(), class_7995.class_7996.field_41664);
        public static final class_9139<ByteBuf, SyncType> STREAM_CODEC = class_9135.method_56375(BY_ID, Enum::ordinal);
    }
}
