/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.screen;

import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import me.shedaniel.clothconfig2.ClothConfigInitializer;
import me.shedaniel.clothconfig2.api.scroll.ScrollingContainer;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.DisplayRenderer;
import me.shedaniel.rei.api.client.gui.widgets.*;
import me.shedaniel.rei.api.client.registry.category.ButtonArea;
import me.shedaniel.rei.api.client.registry.category.CategoryRegistry;
import me.shedaniel.rei.api.client.registry.display.DisplayCategory;
import me.shedaniel.rei.api.client.registry.display.DisplayRegistry;
import me.shedaniel.rei.api.client.view.ViewSearchBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.impl.client.REIRuntimeImpl;
import me.shedaniel.rei.impl.client.gui.widget.EntryWidget;
import me.shedaniel.rei.impl.client.gui.widget.InternalWidgets;
import me.shedaniel.rei.impl.display.DisplaySpec;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;

@ApiStatus.Internal
public class CompositeDisplayViewingScreen extends AbstractDisplayViewingScreen {
    private final List<Widget> widgets = Lists.newArrayList();
    private final List<Button> buttonList = Lists.newArrayList();
    private final List<DisplayRenderer> displayRenderers = Lists.newArrayList();
    public Rectangle scrollListBounds;
    private int selectedRecipeIndex = 0;
    private final ScrollingContainer scrolling = new ScrollingContainer() {
        @Override
        public Rectangle getBounds() {
            return new Rectangle(scrollListBounds.x + 1, scrollListBounds.y + 1, scrollListBounds.width - 2, scrollListBounds.height - 2);
        }
        
        @Override
        public int getMaxScrollHeight() {
            int i = 0;
            for (Button button : buttonList) {
                i += button.getBounds().height;
            }
            return i;
        }
    };
    private float scrollBarAlpha = 0;
    private float scrollBarAlphaFuture = 0;
    private long scrollBarAlphaFutureTime = -1;
    
    public CompositeDisplayViewingScreen(Map<DisplayCategory<?>, List<DisplaySpec>> categoryMap, @Nullable CategoryIdentifier<?> category) {
        super(categoryMap, category);
    }
    
    @Override
    public void recalculateCategoryPage() {
        super.recalculateCategoryPage();
        this.selectedRecipeIndex = 0;
    }
    
    @Override
    public void method_25426() {
        super.method_25426();
        this.method_25396().clear();
        this.widgets.clear();
        this.buttonList.clear();
        this.displayRenderers.clear();
        int largestWidth = field_22789 - 100;
        int largestHeight = field_22790 - 40;
        DisplayCategory<Display> category = getCurrentCategory();
        DisplaySpec display = categoryMap.get(category).get(selectedRecipeIndex);
        int guiWidth = class_3532.method_15340(category.getDisplayWidth(display.provideInternalDisplay()) + 30, 0, largestWidth) + 100;
        int guiHeight = class_3532.method_15340(category.getDisplayHeight() + 40, 166, largestHeight);
        this.bounds = new Rectangle(field_22789 / 2 - guiWidth / 2, field_22790 / 2 - guiHeight / 2, guiWidth, guiHeight);
        
        List<EntryIngredient> workstations = CategoryRegistry.getInstance().get(category.getCategoryIdentifier()).getWorkstations();
        if (!workstations.isEmpty()) {
            int ww = class_3532.method_15375((bounds.width - 16) / 18f);
            int w = Math.min(ww, workstations.size());
            int h = class_3532.method_15386(workstations.size() / ((float) ww));
            int xx = bounds.x + 16;
            int yy = bounds.y + bounds.height + 2;
            widgets.add(Widgets.createCategoryBase(new Rectangle(xx - 5, bounds.y + bounds.height - 5, 10 + w * 16, 12 + h * 16)));
            widgets.add(Widgets.createSlotBase(new Rectangle(xx - 1, yy - 1, 2 + w * 16, 2 + h * 16)));
            int index = 0;
            for (EntryIngredient workingStation : workstations) {
                widgets.add(new DefaultDisplayViewingScreen.WorkstationSlotWidget(xx, yy, workingStation));
                index++;
                xx += 16;
                if (index >= ww) {
                    index = 0;
                    xx = bounds.x + 16;
                    yy += 16;
                }
            }
        }
        
        this.widgets.add(Widgets.createCategoryBase(bounds));
        this.scrollListBounds = new Rectangle(bounds.x + 4, bounds.y + 17, 97 + 5, guiHeight - 17 - 7);
        this.widgets.add(Widgets.createSlotBase(scrollListBounds));
        
        Rectangle recipeBounds = new Rectangle(bounds.x + 100 + (guiWidth - 100) / 2 - category.getDisplayWidth(display.provideInternalDisplay()) / 2, bounds.y + bounds.height / 2 - category.getDisplayHeight() / 2, category.getDisplayWidth(display.provideInternalDisplay()), category.getDisplayHeight());
        List<Widget> setupDisplay;
        try {
            setupDisplay = getCurrentCategoryView(display.provideInternalDisplay()).setupDisplay(display.provideInternalDisplay(), recipeBounds);
        } catch (Throwable throwable) {
            throwable.printStackTrace();
            setupDisplay = new ArrayList<>();
            setupDisplay.add(Widgets.createRecipeBase(recipeBounds).color(0xFFBB0000));
            setupDisplay.add(Widgets.createLabel(new Point(recipeBounds.getCenterX(), recipeBounds.getCenterY() - 8), class_2561.method_43470("Failed to initiate setupDisplay")));
            setupDisplay.add(Widgets.createLabel(new Point(recipeBounds.getCenterX(), recipeBounds.getCenterY() + 1), class_2561.method_43470("Check console for error")));
        }
        setupTags(setupDisplay);
        transformFiltering(setupDisplay);
        transformIngredientNotice(setupDisplay, ingredientStackToNotice);
        transformResultNotice(setupDisplay, resultStackToNotice);
        unifyIngredients(setupDisplay);
        for (EntryWidget widget : Widgets.<EntryWidget>walk(widgets, EntryWidget.class::isInstance)) {
            widget.removeTagMatch = true;
        }
        this.widgets.addAll(setupDisplay);
        Optional<ButtonArea> supplier = CategoryRegistry.getInstance().get(category.getCategoryIdentifier()).getPlusButtonArea();
        if (supplier.isPresent() && supplier.get().get(recipeBounds) != null)
            this.widgets.add(Widgets.withTranslate(InternalWidgets.createAutoCraftingButtonWidget(recipeBounds, supplier.get().get(recipeBounds), class_2561.method_43470(supplier.get().getButtonText()), display::provideInternalDisplay, display::provideInternalDisplayIds, setupDisplay, category), 0, 0, 100));

        this.initTabs(this.bounds.width);
        this.widgets.addAll(this.tabs.widgets());
        
        int index = 0;
        for (DisplaySpec recipeDisplay : categoryMap.get(category)) {
            int finalIndex = index;
            DisplayRenderer displayRenderer;
            displayRenderers.add(displayRenderer = getCurrentCategoryView(recipeDisplay.provideInternalDisplay()).getDisplayRenderer(recipeDisplay.provideInternalDisplay()));
            buttonList.add(Widgets.createButton(new Rectangle(bounds.x + 5, 0, displayRenderer.getWidth(), displayRenderer.getHeight()), class_2561.method_43473())
                    .onClick(button -> {
                        selectedRecipeIndex = finalIndex;
                        CompositeDisplayViewingScreen.this.method_25426();
                    })
                    .containsMousePredicate((button, point) -> {
                        return (button.getBounds().contains(point) && scrollListBounds.contains(point)) || button.method_25370();
                    })
                    .onRender((matrices, button) -> button.setEnabled(selectedRecipeIndex != finalIndex)));
            index++;
        }
        
        this.widgets.add(Widgets.createClickableLabel(new Point(bounds.x + 4 + scrollListBounds.width / 2, bounds.y + 6), categories.get(selectedCategoryIndex).getTitle(), label -> {
            ViewSearchBuilder.builder().addAllCategories().open();
        }).tooltip(class_2561.method_43471("text.rei.view_all_categories"), class_2561.method_43469("text.rei.view_all_categories.tooltip", CategoryRegistry.getInstance().stream().filter(config -> !DisplayRegistry.getInstance().get(config.getCategoryIdentifier()).isEmpty()).count()).method_27692(class_124.field_1063)).noShadow().color(0xFF404040, 0xFFBBBBBB).hoveredColor(0xFF0041FF, 0xFFFFBD4D));
        
        this.widgets.add(new ButtonListWidget(buttonList));
        this.method_25396().addAll(widgets);
    }
    
    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        if (scrolling.updateDraggingState(mouseX, mouseY, button)) {
            scrollBarAlpha = 1;
            return true;
        }
        if (ConfigObject.getInstance().getNextPageKeybind().matchesMouse(button)) {
            selectedRecipeIndex++;
            if (selectedRecipeIndex >= categoryMap.get(categories.get(selectedCategoryIndex)).size())
                selectedRecipeIndex = 0;
            method_25426();
            return true;
        } else if (ConfigObject.getInstance().getPreviousPageKeybind().matchesMouse(button)) {
            selectedRecipeIndex--;
            if (selectedRecipeIndex < 0)
                selectedRecipeIndex = categoryMap.get(categories.get(selectedCategoryIndex)).size() - 1;
            method_25426();
            return true;
        } else if (ConfigObject.getInstance().getPreviousScreenKeybind().matchesMouse(button)) {
            if (REIRuntimeImpl.getInstance().hasLastDisplayScreen()) {
                field_22787.method_1507(REIRuntimeImpl.getInstance().getLastDisplayScreen());
            } else {
                field_22787.method_1507(REIRuntime.getInstance().getPreviousScreen());
            }
            return true;
        }
        for (class_364 entry : method_25396())
            if (entry.method_25402(mouseX, mouseY, button)) {
                method_25395(entry);
                if (button == 0)
                    method_25398(true);
                return true;
            }
        return super.method_25402(mouseX, mouseY, button);
    }
    
    @Override
    public Optional<class_364> method_19355(double mouseX, double mouseY) {
        // Reverse iteration
        for (int i = widgets.size() - 1; i >= 0; i--) {
            Widget widget = widgets.get(i);
            if (widget.containsMouse(mouseX, mouseY)) {
                return Optional.of(widget);
            }
        }
        
        return Optional.empty();
    }
    
    @Override
    public boolean method_25400(char char_1, int int_1) {
        for (class_364 listener : method_25396())
            if (listener.method_25400(char_1, int_1))
                return true;
        return super.method_25400(char_1, int_1);
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        double height = scrolling.getMaxScrollHeight();
        if (scrollListBounds.contains(mouseX, mouseY) && height > scrollListBounds.height - 2 && amountY != 0) {
            scrolling.offset(ClothConfigInitializer.getScrollStep() * -amountY, true);
            if (scrollBarAlphaFuture == 0)
                scrollBarAlphaFuture = 1f;
            if (System.currentTimeMillis() - scrollBarAlphaFutureTime > 300f)
                scrollBarAlphaFutureTime = System.currentTimeMillis();
            return true;
        }
        REIRuntimeImpl.isWithinRecipeViewingScreen = true;
        for (class_364 listener : method_25396()) {
            if (listener.method_25401(mouseX, mouseY, amountX, amountY)) {
                REIRuntimeImpl.isWithinRecipeViewingScreen = false;
                return true;
            }
        }
        REIRuntimeImpl.isWithinRecipeViewingScreen = false;
        if (bounds.contains(PointHelper.ofMouse()) && amountY != 0) {
            if (amountY < 0 && categoryMap.get(categories.get(selectedCategoryIndex)).size() > 1) {
                selectedRecipeIndex++;
                if (selectedRecipeIndex >= categoryMap.get(categories.get(selectedCategoryIndex)).size())
                    selectedRecipeIndex = 0;
                method_25426();
                return true;
            } else if (categoryMap.get(categories.get(selectedCategoryIndex)).size() > 1) {
                selectedRecipeIndex--;
                if (selectedRecipeIndex < 0)
                    selectedRecipeIndex = categoryMap.get(categories.get(selectedCategoryIndex)).size() - 1;
                method_25426();
                return true;
            }
        }
        return super.method_25401(mouseX, mouseY, amountX, amountY);
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        this.widgets.sort(Comparator.comparingDouble(Widget::getZRenderingPriority));
        if (ConfigObject.getInstance().isCompositeScrollBarPermanent()) {
            scrollBarAlphaFutureTime = System.currentTimeMillis();
            scrollBarAlphaFuture = 0;
            scrollBarAlpha = 1;
        } else if (scrollBarAlphaFutureTime > 0) {
            long l = System.currentTimeMillis() - scrollBarAlphaFutureTime;
            if (l > 300f) {
                if (scrollBarAlphaFutureTime == 0) {
                    scrollBarAlpha = scrollBarAlphaFuture;
                    scrollBarAlphaFutureTime = -1;
                } else if (l > 2000f && scrollBarAlphaFuture == 1) {
                    scrollBarAlphaFuture = 0;
                    scrollBarAlphaFutureTime = System.currentTimeMillis();
                } else
                    scrollBarAlpha = scrollBarAlphaFuture;
            } else {
                if (scrollBarAlphaFuture == 0)
                    scrollBarAlpha = Math.min(scrollBarAlpha, 1 - Math.min(1f, l / 300f));
                else if (scrollBarAlphaFuture == 1)
                    scrollBarAlpha = Math.max(Math.min(1f, l / 300f), scrollBarAlpha);
            }
        }
        scrolling.updatePosition(delta);
        super.method_25394(graphics, mouseX, mouseY, delta);
        getOverlay().method_25394(graphics, mouseX, mouseY, delta);
        for (Widget widget : widgets) {
            RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
            widget.method_25394(graphics, mouseX, mouseY, delta);
        }
    }
    
    @Override
    public boolean method_25406(double double_1, double double_2, int int_1) {
        for (class_364 entry : method_25396())
            if (entry.method_25406(double_1, double_2, int_1))
                return true;
        return super.method_25406(double_1, double_2, int_1);
    }
    
    @Override
    public boolean method_25403(double mouseX, double mouseY, int int_1, double double_3, double double_4) {
        if (scrolling.mouseDragged(mouseX, mouseY, int_1, double_3, double_4)) {
            scrollBarAlphaFutureTime = System.currentTimeMillis();
            scrollBarAlphaFuture = 1f;
            return true;
        }
        for (class_364 entry : method_25396())
            if (entry.method_25403(mouseX, mouseY, int_1, double_3, double_4))
                return true;
        return super.method_25403(mouseX, mouseY, int_1, double_3, double_4);
    }
    
    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (ConfigObject.getInstance().getNextPageKeybind().matchesKey(keyCode, scanCode)) {
            if (categoryMap.get(categories.get(selectedCategoryIndex)).size() > 1) {
                selectedRecipeIndex++;
                if (selectedRecipeIndex >= categoryMap.get(categories.get(selectedCategoryIndex)).size())
                    selectedRecipeIndex = 0;
                method_25426();
                return true;
            }
            return false;
        } else if (ConfigObject.getInstance().getPreviousPageKeybind().matchesKey(keyCode, scanCode)) {
            if (categoryMap.get(categories.get(selectedCategoryIndex)).size() > 1) {
                selectedRecipeIndex--;
                if (selectedRecipeIndex < 0)
                    selectedRecipeIndex = categoryMap.get(categories.get(selectedCategoryIndex)).size() - 1;
                method_25426();
                return true;
            }
            return false;
        }
        for (class_364 element : method_25396())
            if (element.method_25404(keyCode, scanCode, modifiers))
                return true;
        if (keyCode == 256) {
            class_310.method_1551().method_1507(REIRuntime.getInstance().getPreviousScreen());
            return true;
        }
        return super.method_25404(keyCode, scanCode, modifiers);
    }
    
    private class ButtonListWidget extends Widget {
        private final List<Button> buttonList;
        
        public ButtonListWidget(List<Button> buttonList) {
            this.buttonList = buttonList;
        }
        
        @Override
        public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
            int yOffset = 0;
            graphics.method_51448().method_22903();
            graphics.method_44379(scrolling.getBounds().x, scrolling.getBounds().y, scrolling.getBounds().getMaxX(), scrolling.getBounds().getMaxY());
            for (Button button : buttonList) {
                button.getBounds().y = scrollListBounds.y + 1 + yOffset - scrolling.scrollAmountInt();
                if (button.getBounds().getMaxY() > scrollListBounds.getMinY() && button.getBounds().getMinY() < scrollListBounds.getMaxY()) {
                    button.method_25394(graphics, mouseX, mouseY, delta);
                }
                yOffset += button.getBounds().height;
            }
            for (int i = 0; i < buttonList.size(); i++) {
                if (buttonList.get(i).getBounds().getMaxY() > scrollListBounds.getMinY() && buttonList.get(i).getBounds().getMinY() < scrollListBounds.getMaxY()) {
                    displayRenderers.get(i).render(graphics, buttonList.get(i).getBounds(), mouseX, mouseY, delta);
                    Optional.ofNullable(displayRenderers.get(i).getTooltip(TooltipContext.of(new Point(mouseX, mouseY), class_1792.class_9635.method_59528(minecraft.field_1687)))).ifPresent(Tooltip::queue);
                }
            }
            scrolling.renderScrollBar(graphics, 0, scrollBarAlpha);
            graphics.method_44380();
            graphics.method_51448().method_22909();
        }
        
        @Override
        public List<? extends class_364> method_25396() {
            return buttonList;
        }
    }
}
