/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.config;

import com.google.common.base.Preconditions;
import dev.architectury.utils.value.IntValue;
import me.shedaniel.clothconfig2.api.Modifier;
import me.shedaniel.clothconfig2.api.ModifierKeyCode;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.gui.widgets.Widget;
import me.shedaniel.rei.api.client.gui.widgets.Widgets;
import me.shedaniel.rei.api.client.overlay.ScreenOverlay;
import me.shedaniel.rei.api.client.registry.entry.EntryRegistry;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.impl.client.REIRuntimeImpl;
import me.shedaniel.rei.impl.client.config.ConfigManagerImpl;
import me.shedaniel.rei.impl.client.config.ConfigObjectImpl;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.config.components.ConfigCategoriesListWidget;
import me.shedaniel.rei.impl.client.gui.config.components.ConfigEntriesListWidget;
import me.shedaniel.rei.impl.client.gui.config.components.ConfigSearchListWidget;
import me.shedaniel.rei.impl.client.gui.config.options.AllREIConfigCategories;
import me.shedaniel.rei.impl.client.gui.config.options.CompositeOption;
import me.shedaniel.rei.impl.client.gui.config.options.OptionCategory;
import me.shedaniel.rei.impl.client.gui.config.options.OptionGroup;
import me.shedaniel.rei.impl.client.gui.modules.Menu;
import me.shedaniel.rei.impl.client.gui.widget.HoleWidget;
import me.shedaniel.rei.impl.client.gui.widget.basewidgets.TextFieldWidget;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_364;
import net.minecraft.class_3675;
import net.minecraft.class_437;
import org.apache.commons.lang3.mutable.Mutable;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;

import static me.shedaniel.rei.impl.client.gui.config.options.ConfigUtils.literal;
import static me.shedaniel.rei.impl.client.gui.config.options.ConfigUtils.translatable;

public class REIConfigScreen extends class_437 implements ConfigAccess {
    private final class_437 parent;
    private final List<OptionCategory> categories;
    private final List<Widget> widgets = new ArrayList<>();
    private final Map<String, ?> defaultOptions = new HashMap<>();
    private final Map<String, ?> options = new HashMap<>();
    private OptionCategory activeCategory;
    private boolean searching;
    @Nullable
    private Menu menu;
    @Nullable
    private Widget menuWidget;
    @Nullable
    private CompositeOption<ModifierKeyCode> focusedKeycodeOption = null;
    private ModifierKeyCode partialKeycode = null;
    
    public REIConfigScreen(class_437 parent) {
        this(parent, AllREIConfigCategories.CATEGORIES);
    }
    
    public REIConfigScreen(class_437 parent, List<OptionCategory> categories) {
        super(class_2561.method_43471("config.roughlyenoughitems.title"));
        this.parent = parent;
        this.categories = CollectionUtils.map(categories, OptionCategory::copy);
        this.cleanRequiresLevel();
        Preconditions.checkArgument(!this.categories.isEmpty(), "Categories cannot be empty!");
        this.activeCategory = this.categories.get(0);
        
        ConfigObjectImpl defaultConfig = new ConfigObjectImpl();
        ConfigObjectImpl config = ConfigManagerImpl.getInstance().getConfig();
        for (OptionCategory category : this.categories) {
            for (OptionGroup group : category.getGroups()) {
                for (CompositeOption<?> option : group.getOptions()) {
                    ((Map<String, Object>) this.defaultOptions).put(option.getId(), option.getBind().apply(defaultConfig));
                    ((Map<String, Object>) this.options).put(option.getId(), option.getBind().apply(config));
                }
            }
        }
    }
    
    private void cleanRequiresLevel() {
        if (!(REIRuntime.getInstance().getPreviousContainerScreen() == null || class_310.method_1551().method_1562() == null)) {
            return;
        }
        
        for (OptionCategory category : this.categories) {
            for (OptionGroup group : category.getGroups()) {
                group.getOptions().replaceAll(option -> {
                    if (option.isRequiresLevel()) {
                        return new CompositeOption<>(option.getId(), option.getName(), option.getDescription(), i -> 0, (i, v) -> new Object())
                                .entry(value -> translatable("config.rei.texts.requires_level").method_27692(class_124.field_1061))
                                .defaultValue(() -> 1);
                    } else {
                        return option;
                    }
                });
            }
        }
    }
    
    @Override
    public void method_25426() {
        super.method_25426();
        this.widgets.clear();
        this.widgets.add(Widgets.createLabel(new Point(field_22789 / 2, 12), this.field_22785));
        int sideWidth = (int) Math.round(field_22789 / 4.2);
        if (this.searching) {
            this.widgets.add(Widgets.createButton(new Rectangle(8, 32, sideWidth, 20), literal("↩ ").method_10852(translatable("gui.back")))
                    .onClick(button -> setSearching(false)));
            this.widgets.add(HoleWidget.createMenuBackground(new Rectangle(8 + sideWidth + 4, 32, field_22789 - 16 - sideWidth - 4, 20)));
            TextFieldWidget textField = new TextFieldWidget(new Rectangle(8 + sideWidth + 4 + 6, 32 + 6, field_22789 - 16 - sideWidth - 4 - 10, 12)) {
                @Override
                protected void renderSuggestion(class_332 graphics, int x, int y) {
                    int color;
                    if (containsMouse(PointHelper.ofMouse()) || method_25370()) {
                        color = 0xddeaeaea;
                    } else {
                        color = -6250336;
                    }
                    graphics.method_25303(this.font, this.font.method_27523(this.getSuggestion(), this.getWidth()), x, y, color);
                }
            };
            textField.setHasBorder(false);
            textField.setMaxLength(9000);
            this.widgets.add(textField);
            this.widgets.add(Widgets.createDrawableWidget((graphics, mouseX, mouseY, delta) -> {
                textField.setSuggestion(!textField.method_25370() && textField.getText().isEmpty() ? class_1074.method_4662("config.rei.texts.search_options") : null);
                if (!textField.method_25370()) return;
                Rectangle bounds = textField.getBounds();
                graphics.method_25294(bounds.x - 6, bounds.y - 6, bounds.getMaxX() + 4, bounds.y - 5, 0xffe0e0e0);
                graphics.method_25294(bounds.x - 6, bounds.getMaxY() + 1, bounds.getMaxX() + 4, bounds.getMaxY() + 2, 0xffe0e0e0);
                graphics.method_25294(bounds.x - 6, bounds.y - 6, bounds.x - 7, bounds.getMaxY() + 2, 0xffe0e0e0);
                graphics.method_25294(bounds.getMaxX() + 3, bounds.y - 6, bounds.getMaxX() + 4, bounds.getMaxY() + 2, 0xffe0e0e0);
            }));
            this.widgets.add(ConfigSearchListWidget.create(this, this.categories, textField, new Rectangle(8, 32 + 20 + 4, field_22789 - 16, field_22790 - 32 - (32 + 20 + 4))));
        } else {
            boolean singlePane = field_22789 - 20 - sideWidth <= 330;
            int singleSideWidth = 32 + 6 + 4;
            Mutable<Widget> list = new MutableObject<>(createEntriesList(singlePane, singleSideWidth, sideWidth));
            IntValue selectedCategory = new IntValue() {
                @Override
                public void accept(int index) {
                    REIConfigScreen.this.activeCategory = categories.get(index);
                    list.setValue(createEntriesList(singlePane, singleSideWidth, sideWidth));
                }
                
                @Override
                public int getAsInt() {
                    return categories.indexOf(activeCategory);
                }
            };
            if (!singlePane) {
                this.widgets.add(ConfigCategoriesListWidget.create(new Rectangle(8, 32, sideWidth, field_22790 - 32 - 32), categories, selectedCategory));
            } else {
                this.widgets.add(ConfigCategoriesListWidget.createTiny(new Rectangle(8, 32, singleSideWidth - 4, field_22790 - 32 - 32), categories, selectedCategory));
            }
            this.widgets.add(Widgets.delegate(list::getValue));
        }
        
        this.widgets.add(Widgets.createButton(new Rectangle(field_22789 / 2 - 150 - 10, field_22790 - 26, 150, 20), translatable("gui.cancel")).onClick(button -> {
            class_310.method_1551().method_1507(this.parent);
        }));
        this.widgets.add(Widgets.createButton(new Rectangle(field_22789 / 2 + 10, field_22790 - 26, 150, 20), translatable("gui.done")).onClick(button -> {
            for (OptionCategory optionCategory : this.categories) {
                for (OptionGroup group : optionCategory.getGroups()) {
                    for (CompositeOption<?> option : group.getOptions()) {
                        ((BiConsumer<ConfigObjectImpl, Object>) option.getSave()).accept(ConfigManagerImpl.getInstance().getConfig(), this.get(option));
                    }
                }
            }
            
            ConfigManagerImpl.getInstance().saveConfig();
            EntryRegistry.getInstance().refilter();
            REIRuntime.getInstance().getOverlay().ifPresent(ScreenOverlay::queueReloadOverlay);
            if (REIRuntimeImpl.getSearchField() != null) {
                ScreenOverlayImpl.getEntryListWidget().updateSearch(REIRuntimeImpl.getSearchField().getText(), true);
            }
            class_310.method_1551().method_1507(this.parent);
        }));
    }
    
    private Widget createEntriesList(boolean singlePane, int singleSideWidth, int sideWidth) {
        return ConfigEntriesListWidget.create(this, new Rectangle(singlePane ? 8 + singleSideWidth : 12 + sideWidth, 32, singlePane ? field_22789 - 16 - singleSideWidth : field_22789 - 20 - sideWidth, field_22790 - 32 - 32), activeCategory.getGroups());
    }
    
    public Map<String, ?> getDefaultOptions() {
        return defaultOptions;
    }
    
    public Map<String, ?> getOptions() {
        return options;
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        super.method_25394(graphics, mouseX, mouseY, delta);
        for (Widget widget : widgets) {
            widget.method_25394(graphics, mouseX, mouseY, delta);
        }
        ScreenOverlayImpl.getInstance().lateRender(graphics, mouseX, mouseY, delta);
    }
    
    @Override
    public void method_25419() {
        if (searching) {
            setSearching(false);
        } else {
            this.field_22787.method_1507(this.parent);
        }
    }
    
    @Override
    public List<? extends class_364> method_25396() {
        return (List<? extends class_364>) (List<?>) this.widgets;
    }
    
    @Override
    public boolean method_25400(char character, int modifiers) {
        if (menu != null && menu.method_25400(character, modifiers))
            return true;
        for (class_364 listener : method_25396())
            if (listener.method_25400(character, modifiers))
                return true;
        return super.method_25400(character, modifiers);
    }
    
    @Override
    public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
        if (menu != null && menu.method_25403(mouseX, mouseY, button, deltaX, deltaY))
            return true;
        for (class_364 entry : method_25396())
            if (entry.method_25403(mouseX, mouseY, button, deltaX, deltaY))
                return true;
        return super.method_25403(mouseX, mouseY, button, deltaX, deltaY);
    }
    
    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        if (menu != null) {
            if (!menu.method_25402(mouseX, mouseY, button))
                closeMenu();
            return true;
        }
        
        if (this.focusedKeycodeOption != null && this.partialKeycode != null) {
            if (this.partialKeycode.isUnknown()) {
                this.partialKeycode.setKeyCode(class_3675.class_307.field_1672.method_1447(button));
            } else if (this.partialKeycode.getType() == class_3675.class_307.field_1668) {
                Modifier modifier = this.partialKeycode.getModifier();
                int code = this.partialKeycode.getKeyCode().method_1444();
                if (class_310.field_1703 ? code == 343 || code == 347 : code == 341 || code == 345) {
                    this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), true, modifier.hasShift()));
                    this.partialKeycode.setKeyCode(class_3675.class_307.field_1672.method_1447(button));
                    return true;
                }
                
                if (code == 344 || code == 340) {
                    this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), modifier.hasControl(), true));
                    this.partialKeycode.setKeyCode(class_3675.class_307.field_1672.method_1447(button));
                    return true;
                }
                
                if (code == 342 || code == 346) {
                    this.partialKeycode.setModifier(Modifier.of(true, modifier.hasControl(), modifier.hasShift()));
                    this.partialKeycode.setKeyCode(class_3675.class_307.field_1672.method_1447(button));
                    return true;
                }
            }
            
            return true;
        }
        
        return super.method_25402(mouseX, mouseY, button);
    }
    
    @Override
    public Optional<class_364> method_19355(double mouseX, double mouseY) {
        if (menu != null) {
            if (menu.containsMouse(mouseX, mouseY)) {
                return Optional.of(menu);
            }
        }
        
        // Reverse iteration
        for (int i = widgets.size() - 1; i >= 0; i--) {
            Widget widget = widgets.get(i);
            if (widget.containsMouse(mouseX, mouseY)) {
                return Optional.of(widget);
            }
        }
        
        return Optional.empty();
    }
    
    @Override
    public boolean method_25406(double mouseX, double mouseY, int button) {
        if (menu != null && menu.method_25406(mouseX, mouseY, button))
            return true;
        if (this.focusedKeycodeOption != null && this.partialKeycode != null && !this.partialKeycode.isUnknown()) {
            this.set(this.focusedKeycodeOption, this.partialKeycode);
            this.focusKeycode(null);
            return true;
        }
        for (class_364 entry : method_25396())
            if (entry.method_25406(mouseX, mouseY, button))
                return true;
        return super.method_25406(mouseX, mouseY, button);
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        if (menu != null && menu.method_25401(mouseX, mouseY, amountX, amountY))
            return true;
        for (class_364 listener : method_25396())
            if (listener.method_25401(mouseX, mouseY, amountX, amountY))
                return true;
        return super.method_25401(mouseX, mouseY, amountX, amountY);
    }
    
    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (this.focusedKeycodeOption != null) {
            if (keyCode != 256) {
                if (this.partialKeycode.isUnknown()) {
                    this.partialKeycode.setKeyCode(class_3675.method_15985(keyCode, scanCode));
                } else {
                    Modifier modifier = this.partialKeycode.getModifier();
                    if (this.partialKeycode.getType() == class_3675.class_307.field_1668) {
                        int code = this.partialKeycode.getKeyCode().method_1444();
                        if (class_310.field_1703 ? code == 343 || code == 347 : code == 341 || code == 345) {
                            this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), true, modifier.hasShift()));
                            this.partialKeycode.setKeyCode(class_3675.method_15985(keyCode, scanCode));
                            return true;
                        }
                        
                        if (code == 344 || code == 340) {
                            this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), modifier.hasControl(), true));
                            this.partialKeycode.setKeyCode(class_3675.method_15985(keyCode, scanCode));
                            return true;
                        }
                        
                        if (code == 342 || code == 346) {
                            this.partialKeycode.setModifier(Modifier.of(true, modifier.hasControl(), modifier.hasShift()));
                            this.partialKeycode.setKeyCode(class_3675.method_15985(keyCode, scanCode));
                            return true;
                        }
                    }
                    
                    if (class_310.field_1703 ? keyCode == 343 || keyCode == 347 : keyCode == 341 || keyCode == 345) {
                        this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), true, modifier.hasShift()));
                        return true;
                    }
                    
                    if (keyCode == 344 || keyCode == 340) {
                        this.partialKeycode.setModifier(Modifier.of(modifier.hasAlt(), modifier.hasControl(), true));
                        return true;
                    }
                    
                    if (keyCode == 342 || keyCode == 346) {
                        this.partialKeycode.setModifier(Modifier.of(true, modifier.hasControl(), modifier.hasShift()));
                        return true;
                    }
                }
            } else {
                this.set(this.focusedKeycodeOption, ModifierKeyCode.unknown());
                this.focusKeycode(null);
            }
            
            return true;
        }
        
        return super.method_25404(keyCode, scanCode, modifiers);
    }
    
    @Override
    public boolean method_16803(int keyCode, int scanCode, int modifiers) {
        if (this.focusedKeycodeOption != null && this.partialKeycode != null) {
            this.set(this.focusedKeycodeOption, this.partialKeycode);
            this.focusKeycode(null);
            return true;
        }
        
        return super.method_16803(keyCode, scanCode, modifiers);
    }
    
    @Override
    public void openMenu(Menu menu) {
        if (this.menu != null) {
            this.widgets.remove(this.menuWidget);
        }
        this.menu = menu;
        this.widgets.add(this.menuWidget = Widgets.withTranslate(menu, 0, 0, 300));
    }
    
    @Override
    public void closeMenu() {
        this.widgets.remove(this.menuWidget);
        this.menu = null;
        this.menuWidget = null;
    }
    
    @Override
    public <T> T get(CompositeOption<T> option) {
        return (T) getOptions().get(option.getId());
    }
    
    @Override
    public <T> void set(CompositeOption<T> option, T value) {
        ((Map<String, Object>) getOptions()).put(option.getId(), value);
    }
    
    @Override
    public <T> T getDefault(CompositeOption<T> option) {
        return (T) getDefaultOptions().get(option.getId());
    }
    
    @Override
    public void focusKeycode(CompositeOption<ModifierKeyCode> option) {
        this.focusedKeycodeOption = option;
        
        if (this.focusedKeycodeOption != null) {
            this.partialKeycode = this.get(this.focusedKeycodeOption);
            this.partialKeycode.setKeyCodeAndModifier(class_3675.field_16237, Modifier.none());
        } else {
            this.partialKeycode = null;
        }
    }
    
    @Override
    @Nullable
    public CompositeOption<ModifierKeyCode> getFocusedKeycode() {
        return this.focusedKeycodeOption;
    }
    
    public void setSearching(boolean searching) {
        this.searching = searching;
        this.method_25423(this.field_22787, this.field_22789, this.field_22790);
    }
    
    public boolean isSearching() {
        return searching;
    }
}
