/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.client.entry;

import com.google.common.base.Suppliers;
import com.google.common.collect.Lists;
import com.mojang.serialization.Codec;
import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.client.fluid.ClientFluidStackHooks;
import dev.architectury.hooks.fluid.FluidStackHooks;
import dev.architectury.platform.Platform;
import dev.architectury.utils.Env;
import dev.architectury.utils.EnvExecutor;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.entry.renderer.EntryRenderer;
import me.shedaniel.rei.api.client.gui.widgets.Tooltip;
import me.shedaniel.rei.api.client.gui.widgets.TooltipContext;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.entry.EntrySerializer;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.comparison.ComparisonContext;
import me.shedaniel.rei.api.common.entry.comparison.FluidComparatorRegistry;
import me.shedaniel.rei.api.common.entry.type.EntryDefinition;
import me.shedaniel.rei.api.common.entry.type.EntryType;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1058;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_128;
import net.minecraft.class_129;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3609;
import net.minecraft.class_3611;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.class_9326;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class FluidEntryDefinition implements EntryDefinition<FluidStack>, EntrySerializer<FluidStack> {
    private static final String FLUID_AMOUNT = Platform.isForge() ? "tooltip.rei.fluid_amount.forge" : "tooltip.rei.fluid_amount";
    @Environment(EnvType.CLIENT)
    private EntryRenderer<FluidStack> renderer;
    
    public FluidEntryDefinition() {
        EnvExecutor.runInEnv(Env.CLIENT, () -> () -> Client.init(this));
    }
    
    @Environment(EnvType.CLIENT)
    private static class Client {
        private static void init(FluidEntryDefinition definition) {
            definition.renderer = new FluidEntryRenderer();
        }
    }
    
    @Override
    public Class<FluidStack> getValueType() {
        return FluidStack.class;
    }
    
    @Override
    public EntryType<FluidStack> getType() {
        return VanillaEntryTypes.FLUID;
    }
    
    @Override
    @Environment(EnvType.CLIENT)
    public EntryRenderer<FluidStack> getRenderer() {
        return renderer;
    }
    
    @Override
    @Nullable
    public class_2960 getIdentifier(EntryStack<FluidStack> entry, FluidStack value) {
        return class_7923.field_41173.method_10221(value.getFluid());
    }
    
    @Override
    public boolean isEmpty(EntryStack<FluidStack> entry, FluidStack value) {
        return value.isEmpty();
    }
    
    @Override
    public FluidStack copy(EntryStack<FluidStack> entry, FluidStack value) {
        return value.copy();
    }
    
    @Override
    public FluidStack normalize(EntryStack<FluidStack> entry, FluidStack value) {
        class_3611 fluid = value.getFluid();
        if (fluid instanceof class_3609 flowingFluid) fluid = flowingFluid.method_15751();
        return FluidStack.create(fluid, FluidStack.bucketAmount(), value.getPatch());
    }
    
    @Override
    public FluidStack wildcard(EntryStack<FluidStack> entry, FluidStack value) {
        class_3611 fluid = value.getFluid();
        if (fluid instanceof class_3609) fluid = ((class_3609) fluid).method_15751();
        return FluidStack.create(fluid, FluidStack.bucketAmount());
    }
    
    @Override
    @Nullable
    public class_1799 cheatsAs(EntryStack<FluidStack> entry, FluidStack value) {
        if (value.isEmpty()) return class_1799.field_8037;
        class_1792 bucket = value.getFluid().method_15774();
        if (bucket == null) return class_1799.field_8037;
        return new class_1799(bucket);
    }
    
    @Nullable
    @Override
    public FluidStack add(FluidStack o1, FluidStack o2) {
        return o1.copyWithAmount(o1.getAmount() + o2.getAmount());
    }
    
    @Override
    public long hash(EntryStack<FluidStack> entry, FluidStack value, ComparisonContext context) {
        int code = 1;
        code = 31 * code + value.getFluid().hashCode();
        code = 31 * code + Long.hashCode(FluidComparatorRegistry.getInstance().hashOf(context, value));
        return code;
    }
    
    @Override
    public boolean equals(FluidStack o1, FluidStack o2, ComparisonContext context) {
        if (o1.getFluid() != o2.getFluid())
            return false;
        return FluidComparatorRegistry.getInstance().hashOf(context, o1) == FluidComparatorRegistry.getInstance().hashOf(context, o2);
    }
    
    @Override
    @Nullable
    public EntrySerializer<FluidStack> getSerializer() {
        return this;
    }
    
    @Override
    public boolean acceptsNull() {
        return false;
    }
    
    @Override
    public Codec<FluidStack> codec() {
        return FluidStack.CODEC;
    }
    
    @Override
    public class_9139<class_9129, FluidStack> streamCodec() {
        return FluidStack.STREAM_CODEC;
    }
    
    @Override
    public class_2561 asFormattedText(EntryStack<FluidStack> entry, FluidStack value) {
        return value.getName();
    }
    
    @Override
    public Stream<? extends class_6862<?>> getTagsFor(EntryStack<FluidStack> entry, FluidStack value) {
        return value.getFluid().method_40178().method_40228();
    }
    
    @Override
    public void fillCrashReport(class_128 report, class_129 category, EntryStack<FluidStack> entry) {
        EntryDefinition.super.fillCrashReport(report, category, entry);
        FluidStack stack = entry.getValue();
        category.method_577("Fluid Type", () -> String.valueOf(class_7923.field_41173.method_10221(stack.getFluid())));
        category.method_577("Fluid Amount", () -> String.valueOf(stack.getAmount()));
        category.method_577("Fluid NBT", () -> class_9326.field_49589.encodeStart(BasicDisplay.registryAccess().method_57093(class_2509.field_11560), stack.getPatch()).result().map(class_2520::toString).orElse("Error"));
    }
    
    @Environment(EnvType.CLIENT)
    public static class FluidEntryRenderer implements EntryRenderer<FluidStack> {
        private static final Supplier<class_1058> MISSING_SPRITE = Suppliers.memoize(() -> {
//            TextureAtlas atlas = Minecraft.getInstance().getModelManager().getAtlas(TextureAtlas.LOCATION_BLOCKS);
//            return atlas.getSprite(MissingTextureAtlasSprite.getLocation());
            return null;
        });
        
        private class_1058 missingTexture() {
            return MISSING_SPRITE.get();
        }
        
        @Override
        public void render(EntryStack<FluidStack> entry, class_332 graphics, Rectangle bounds, int mouseX, int mouseY, float delta) {
            FluidStack stack = entry.getValue();
            if (stack.isEmpty()) return;
            class_1058 sprite = ClientFluidStackHooks.getStillTexture(stack);
            if (sprite == null) return;
            int color = ClientFluidStackHooks.getColor(stack);
            
            /*SpriteRenderer.beginPass()
                    .setup(immediate, RenderType.solid())
                    .sprite(sprite)
                    .color(color)
                    .light(0x00f000f0)
                    .overlay(OverlayTexture.NO_OVERLAY)
                    .alpha(0xff)
                    .normal(graphics.pose(), 0, 0, 0)
                    .position(graphics.pose(), bounds.x, bounds.getMaxY() - bounds.height * Mth.clamp(entry.get(EntryStack.Settings.FLUID_RENDER_RATIO), 0, 1), bounds.getMaxX(), bounds.getMaxY(), 0)
                    .next(TextureAtlas.LOCATION_BLOCKS);*/
        }
        
        @Override
        @Nullable
        public Tooltip getTooltip(EntryStack<FluidStack> entry, TooltipContext context) {
            if (entry.isEmpty())
                return null;
            List<class_2561> toolTip = Lists.newArrayList(entry.asFormattedText(context));
            long amount = entry.getValue().getAmount();
            if (amount >= 0 && entry.get(EntryStack.Settings.FLUID_AMOUNT_VISIBLE)) {
                String amountTooltip = class_1074.method_4662(FLUID_AMOUNT, entry.getValue().getAmount());
                if (amountTooltip != null) {
                    toolTip.addAll(Stream.of(amountTooltip.split("\n")).map(class_2561::method_43470).collect(Collectors.toList()));
                }
            }
            if (class_310.method_1551().field_1690.field_1827) {
                class_2960 fluidId = class_7923.field_41173.method_10221(entry.getValue().getFluid());
                toolTip.add((class_2561.method_43470(fluidId.toString())).method_27692(class_124.field_1063));
            }
            return Tooltip.create(toolTip);
        }
    }
}
