/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.config.entries;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import me.shedaniel.clothconfig2.ClothConfigInitializer;
import me.shedaniel.clothconfig2.api.scroll.ScrollingContainer;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.gui.widgets.Tooltip;
import me.shedaniel.rei.api.client.gui.widgets.TooltipContext;
import me.shedaniel.rei.api.client.registry.entry.EntryRegistry;
import me.shedaniel.rei.api.client.search.SearchFilter;
import me.shedaniel.rei.api.client.search.SearchProvider;
import me.shedaniel.rei.api.common.entry.EntrySerializer;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.widget.EntryRendererManager;
import me.shedaniel.rei.impl.client.gui.widget.EntryWidget;
import me.shedaniel.rei.impl.client.gui.widget.UpdatedListWidget;
import me.shedaniel.rei.impl.client.gui.widget.search.OverlaySearchField;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import static me.shedaniel.rei.impl.client.gui.widget.entrylist.EntryListWidget.entrySize;

@ApiStatus.Internal
public class FilteringScreen extends class_437 {
    protected List<EntryStack<?>> selected = Lists.newArrayList();
    protected final ScrollingContainer scrolling = new ScrollingContainer() {
        @Override
        public int getMaxScrollHeight() {
            return class_3532.method_15386(entryStacks.size() / (innerBounds.width / (float) entrySize())) * entrySize() + 28;
        }
        
        @Override
        public Rectangle getBounds() {
            return FilteringScreen.this.getBounds();
        }
        
        @Override
        public int getScrollBarX(int maxX) {
            return field_22789 - 7;
        }
    };
    
    class_437 parent;
    private Set<EntryStack<?>> configFiltered;
    private Tooltip tooltip = null;
    private List<EntryStack<?>> entryStacks = null;
    private Rectangle innerBounds;
    private List<FilteringListEntry> entries = Collections.emptyList();
    private List<class_364> elements = Collections.emptyList();
    
    private record PointPair(Point firstPoint, @Nullable Point secondPoint) {
    }
    
    private List<PointPair> points = new ArrayList<>();
    
    private OverlaySearchField searchField;
    private class_4185 selectAllButton;
    private class_4185 selectNoneButton;
    private class_4185 hideButton;
    private class_4185 showButton;
    private class_4185 backButton;
    private Predicate<Rectangle> selectionCache;
    
    private SearchFilter lastFilter = SearchFilter.matchAll();
    
    public FilteringScreen(Set<EntryStack<?>> configFiltered) {
        super(class_2561.method_43471("config.roughlyenoughitems.filteringScreen"));
        this.configFiltered = configFiltered;
        this.searchField = new OverlaySearchField(0, 0, 0, 0);
        {
            class_2561 selectAllText = class_2561.method_43471("config.roughlyenoughitems.filteredEntries.selectAll");
            this.selectAllButton = class_4185.method_46430(selectAllText, button -> {
                        this.points.clear();
                        this.points.add(new PointPair(new Point(-Integer.MAX_VALUE / 2, -Integer.MAX_VALUE / 2), new Point(Integer.MAX_VALUE / 2, Integer.MAX_VALUE / 2)));
                    })
                    .method_46434(0, 0, class_310.method_1551().field_1772.method_27525(selectAllText) + 10, 20)
                    .method_46431();
        }
        {
            class_2561 selectNoneText = class_2561.method_43471("config.roughlyenoughitems.filteredEntries.selectNone");
            this.selectNoneButton = class_4185.method_46430(selectNoneText, button -> {
                        this.points.clear();
                    })
                    .method_46434(0, 0, class_310.method_1551().field_1772.method_27525(selectNoneText) + 10, 20)
                    .method_46431();
        }
        {
            class_2561 hideText = class_2561.method_43471("config.roughlyenoughitems.filteredEntries.hide");
            this.hideButton = class_4185.method_46430(hideText, button -> {
                        for (int i = 0; i < entryStacks.size(); i++) {
                            EntryStack<?> stack = entryStacks.get(i);
                            FilteringListEntry entry = entries.get(i);
                            entry.getBounds().y = entry.backupY - scrolling.scrollAmountInt();
                            if (entry.isSelected() && !entry.isFiltered()) {
                                configFiltered.add(stack);
                                entry.dirty = true;
                            }
                        }
                    })
                    .method_46434(0, 0, class_310.method_1551().field_1772.method_27525(hideText) + 10, 20)
                    .method_46431();
        }
        {
            class_2561 showText = class_2561.method_43471("config.roughlyenoughitems.filteredEntries.show");
            this.showButton = class_4185.method_46430(showText, button -> {
                        for (int i = 0; i < entryStacks.size(); i++) {
                            EntryStack<?> stack = entryStacks.get(i);
                            FilteringListEntry entry = entries.get(i);
                            entry.getBounds().y = entry.backupY - scrolling.scrollAmountInt();
                            if (entry.isSelected() && configFiltered.remove(stack)) {
                                entry.dirty = true;
                            }
                        }
                    })
                    .method_46434(0, 0, class_310.method_1551().field_1772.method_27525(showText) + 10, 20)
                    .method_46431();
        }
        {
            class_2561 backText = class_2561.method_43470("↩ ").method_10852(class_2561.method_43471("gui.back"));
            this.backButton = class_4185.method_46430(backText, button -> {
                        field_22787.method_1507(parent);
                        this.parent = null;
                    })
                    .method_46434(0, 0, class_310.method_1551().field_1772.method_27525(backText) + 10, 20)
                    .method_46431();
        }
        this.searchField.isMain = false;
    }
    
    @Override
    public void method_25419() {
        this.field_22787.method_1507(parent);
        this.parent = null;
    }
    
    private static Rectangle updateInnerBounds(Rectangle bounds) {
        int width = Math.max(class_3532.method_15375((bounds.width - 2 - 6) / (float) entrySize()), 1);
        return new Rectangle((int) (bounds.getCenterX() - width * entrySize() / 2f), bounds.y + 5, width * entrySize(), bounds.height);
    }
    
    public Rectangle getBounds() {
        return new Rectangle(0, 30, field_22789, this.field_22790 - 30);
    }
    
    @Override
    public void method_25426() {
        super.method_25426();
        Rectangle bounds = getBounds();
        updateSearch(this.searchField.getText());
        this.selectAllButton.method_46421(2);
        this.selectAllButton.method_46419(bounds.getMaxY() - 22);
        this.selectNoneButton.method_46421(4 + selectAllButton.method_25368());
        this.selectNoneButton.method_46419(bounds.getMaxY() - 22);
        int searchFieldWidth = Math.max(bounds.width - (selectNoneButton.method_46426() + selectNoneButton.method_25368() + hideButton.method_25368() + showButton.method_25368() + 12), 100);
        this.searchField.getBounds().setBounds(selectNoneButton.method_46426() + selectNoneButton.method_25368() + 4, bounds.getMaxY() - 21, searchFieldWidth, 18);
        this.hideButton.method_46421(bounds.getMaxX() - hideButton.method_25368() - showButton.method_25368() - 4);
        this.hideButton.method_46419(bounds.getMaxY() - 22);
        this.showButton.method_46421(bounds.getMaxX() - showButton.method_25368() - 2);
        this.showButton.method_46419(bounds.getMaxY() - 22);
        this.backButton.method_46421(4);
        this.backButton.method_46419(4);
        this.searchField.setResponder(this::updateSearch);
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        super.method_25394(graphics, mouseX, mouseY, delta);
        updateSelectionCache();
        Rectangle bounds = getBounds();
        tooltip = null;
        UpdatedListWidget.renderAs(field_22787, field_22789, field_22790, bounds.y, field_22790, graphics, delta);
        if (bounds.isEmpty())
            return;
        graphics.method_44379(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY());
        for (FilteringListEntry entry : entries)
            entry.clearStacks();
        int skip = Math.max(0, class_3532.method_15357(scrolling.scrollAmount() / (float) entrySize()));
        int nextIndex = skip * innerBounds.width / entrySize();
        int i = nextIndex;
        EntryRendererManager<FilteringListEntry> manager = new EntryRendererManager<>();
        for (; i < entryStacks.size(); i++) {
            EntryStack<?> stack = entryStacks.get(i);
            FilteringListEntry entry = entries.get(nextIndex);
            entry.getBounds().y = entry.backupY - scrolling.scrollAmountInt();
            if (entry.getBounds().y > bounds.getMaxY())
                break;
            entry.entry(stack);
            manager.add(entry);
            nextIndex++;
        }
        manager.render(graphics, mouseX, mouseY, delta);
        updatePosition(delta);
        scrolling.renderScrollBar(graphics, 0, REIRuntime.getInstance().isDarkThemeEnabled() ? 0.8F : 1F);
        this.searchField.method_25394(graphics, mouseX, mouseY, delta);
        this.selectAllButton.method_25394(graphics, mouseX, mouseY, delta);
        this.selectNoneButton.method_25394(graphics, mouseX, mouseY, delta);
        this.hideButton.method_25394(graphics, mouseX, mouseY, delta);
        this.showButton.method_25394(graphics, mouseX, mouseY, delta);
        
        graphics.method_44380();
        // TODO: add back border
        /*graphics.drawSpecial(source -> {
            Matrix4f matrix = graphics.pose().last().pose();
            VertexConsumer buffer = source.getBuffer(RenderType.gui());
            buffer.addVertex(matrix, 0, bounds.y + 4, 0.0F).setColor(0, 0, 0, 0);
            buffer.addVertex(matrix, width, bounds.y + 4, 0.0F).setColor(0, 0, 0, 0);
            buffer.addVertex(matrix, width, bounds.y, 0.0F).setColor(0, 0, 0, 255);
            buffer.addVertex(matrix, 0, bounds.y, 0.0F).setColor(0, 0, 0, 255);
        });*/
        
        this.backButton.method_25394(graphics, mouseX, mouseY, delta);
        
        if (tooltip != null) {
            ((ScreenOverlayImpl) REIRuntime.getInstance().getOverlay().get()).renderTooltip(graphics, tooltip);
        }
        
        graphics.method_35720(this.field_22793, this.field_22785.method_30937(), (int) (this.field_22789 / 2.0F - this.field_22793.method_27525(this.field_22785) / 2.0F), 12, -1);
        class_2561 hint = class_2561.method_43471("config.roughlyenoughitems.filteringRulesScreen.hint").method_27692(class_124.field_1054);
        graphics.method_27535(this.field_22793, hint, this.field_22789 - this.field_22793.method_27525(hint) - 15, 12, -1);
    }
    
    private Predicate<Rectangle> getSelection() {
        return selectionCache;
    }
    
    private void updateSelectionCache() {
        if (!points.isEmpty()) {
            Predicate<Rectangle> predicate = rect -> false;
            for (PointPair pair : points) {
                Point firstPoint = pair.firstPoint();
                Point secondPoint = pair.secondPoint();
                if (secondPoint == null) {
                    secondPoint = PointHelper.ofMouse();
                    secondPoint.translate(0, scrolling.scrollAmountInt());
                }
                int left = Math.min(firstPoint.x, secondPoint.x);
                int top = Math.min(firstPoint.y, secondPoint.y);
                int right = Math.max(firstPoint.x, secondPoint.x);
                int bottom = Math.max(firstPoint.y, secondPoint.y);
                Rectangle rectangle = new Rectangle(left, top - scrolling.scrollAmountInt(), Math.max(1, right - left), Math.max(1, bottom - top));
                predicate = predicate.or(rectangle::intersects);
            }
            selectionCache = predicate;
            return;
        }
        selectionCache = rect -> false;
    }
    
    @Override
    public boolean method_25403(class_11909 event, double dx, double dy) {
        if (scrolling.mouseDragged(event.comp_4798(), event.comp_4799(), event.method_74245(), dx, dy))
            return true;
        return super.method_25403(event, dx, dy);
    }
    
    private void updatePosition(float delta) {
        scrolling.updatePosition(delta);
    }
    
    public void updateSearch(String searchTerm) {
        lastFilter = SearchProvider.getInstance().createFilter(searchTerm);
        Set<EntryStack<?>> list = Sets.newLinkedHashSet();
        EntryRegistry.getInstance().getEntryStacks().parallel().filter(this::matches).map(EntryStack::normalize).forEachOrdered(list::add);
        
        entryStacks = Lists.newArrayList(list);
        updateEntriesPosition();
    }
    
    public boolean matches(EntryStack<?> stack) {
        EntrySerializer<?> serializer = stack.getDefinition().getSerializer();
        if (serializer == null) {
            return false;
        }
        return lastFilter.test(stack);
    }
    
    public void updateEntriesPosition() {
        int entrySize = entrySize();
        this.innerBounds = updateInnerBounds(getBounds());
        int width = innerBounds.width / entrySize;
        int pageHeight = innerBounds.height / entrySize;
        int slotsToPrepare = Math.max(entryStacks.size() * 3, width * pageHeight * 3);
        int currentX = 0;
        int currentY = 0;
        List<FilteringListEntry> entries = Lists.newArrayList();
        for (int i = 0; i < slotsToPrepare; i++) {
            int xPos = currentX * entrySize + innerBounds.x;
            int yPos = currentY * entrySize + innerBounds.y;
            entries.add(new FilteringListEntry(xPos, yPos, entrySize));
            currentX++;
            if (currentX >= width) {
                currentX = 0;
                currentY++;
            }
        }
        this.entries = entries;
        this.elements = Lists.newArrayList(entries);
        this.elements.add(searchField);
    }
    
    @Override
    public List<? extends class_364> method_25396() {
        return elements;
    }
    
    @Override
    public boolean method_25402(class_11909 event, boolean doubleClick) {
        if (scrolling.updateDraggingState(event.comp_4798(), event.comp_4799(), event.method_74245()))
            return true;
        
        if (getBounds().contains(event.comp_4798(), event.comp_4799())) {
            if (searchField.method_25402(event, doubleClick)) {
                this.points.clear();
                return true;
            } else if (selectAllButton.method_25402(event, doubleClick)) {
                return true;
            } else if (selectNoneButton.method_25402(event, doubleClick)) {
                return true;
            } else if (hideButton.method_25402(event, doubleClick)) {
                return true;
            } else if (showButton.method_25402(event, doubleClick)) {
                return true;
            } else if (event.method_74245() == 0) {
                if (!class_310.method_1551().method_74187()) {
                    this.points.clear();
                }
                this.points.add(new PointPair(new Point(event.comp_4798(), event.comp_4799() + scrolling.scrollAmount()), null));
                return true;
            }
        }
        return backButton.method_25402(event, doubleClick);
    }
    
    @Override
    public boolean method_25406(class_11909 event) {
        if (event.method_74245() == 0 && !points.isEmpty()) {
            PointPair pair = this.points.get(points.size() - 1);
            if (pair.secondPoint() == null) {
                this.points.set(points.size() - 1, new PointPair(pair.firstPoint(), new Point(event.comp_4798(), event.comp_4799() + scrolling.scrollAmount())));
                return true;
            }
        }
        return super.method_25406(event);
    }
    
    @Override
    public boolean method_25400(class_11905 event) {
        for (class_364 element : method_25396())
            if (element.method_25400(event))
                return true;
        return super.method_25400(event);
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        for (class_364 element : method_25396())
            if (element.method_25404(event))
                return true;
        if (event.method_74241()) {
            this.points.clear();
            this.points.add(new PointPair(new Point(-Integer.MAX_VALUE / 2, -Integer.MAX_VALUE / 2), new Point(Integer.MAX_VALUE / 2, Integer.MAX_VALUE / 2)));
            return true;
        }
        if (event.comp_4795() == 256 && this.method_25422()) {
            this.backButton.method_25306(event);
            return true;
        }
        return false;
    }
    
    public void updateArea(@Nullable String searchTerm) {
        if (searchTerm != null)
            updateSearch(searchTerm);
        else if (entryStacks == null)
            updateSearch("");
        else
            updateEntriesPosition();
    }
    
    @Override
    public boolean method_25401(double double_1, double double_2, double amountX, double amountY) {
        if (getBounds().contains(double_1, double_2) && amountY != 0) {
            scrolling.offset(ClothConfigInitializer.getScrollStep() * -amountY, true);
            return true;
        }
        super.method_25401(double_1, double_2, amountX, amountY);
        return true;
    }
    
    private class FilteringListEntry extends EntryWidget {
        private int backupY;
        private boolean filtered = false;
        private boolean dirty = true;
        
        private FilteringListEntry(int x, int y, int entrySize) {
            super(new Point(x, y));
            this.backupY = y;
            getBounds().width = getBounds().height = entrySize;
            interactableFavorites(false);
            interactable(false);
            noHighlight();
        }
        
        @Override
        public boolean containsMouse(double mouseX, double mouseY) {
            return super.containsMouse(mouseX, mouseY) && FilteringScreen.this.getBounds().contains(mouseX, mouseY);
        }
        
        @Override
        protected void drawExtra(class_332 graphics, int mouseX, int mouseY, float delta) {
            if (isSelected()) {
                boolean filtered = isFiltered();
                Rectangle bounds = getBounds();
                graphics.method_25296(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY(), filtered ? 0x90ffffff : 0x55ffffff, filtered ? 0x90ffffff : 0x55ffffff);
            }
        }
        
        public boolean isSelected() {
            return getSelection().test(getBounds());
        }
        
        public boolean isFiltered() {
            if (dirty) {
                filtered = configFiltered.contains(getCurrentEntry());
                dirty = false;
            }
            return filtered;
        }
        
        @Override
        protected void drawBackground(class_332 graphics, int mouseX, int mouseY, float delta) {
            if (isFiltered()) {
                Rectangle bounds = getBounds();
                graphics.method_25296(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY(), 0xffff0000, 0xffff0000);
            }
        }
        
        @Override
        protected void queueTooltip(class_332 graphics, int mouseX, int mouseY, float delta) {
            if (searchField.containsMouse(mouseX, mouseY))
                return;
            Tooltip tooltip = getCurrentTooltip(TooltipContext.of(new Point(mouseX, mouseY), class_1792.class_9635.method_59528(minecraft.field_1687)));
            if (tooltip != null) {
                FilteringScreen.this.tooltip = tooltip;
            }
        }
    }
}
