/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.common;

import dev.architectury.event.CompoundEventResult;
import dev.architectury.hooks.fluid.FluidBucketHooks;
import dev.architectury.hooks.fluid.FluidStackHooks;
import me.shedaniel.rei.api.common.display.DisplaySerializerRegistry;
import me.shedaniel.rei.api.common.entry.comparison.ItemComparatorRegistry;
import me.shedaniel.rei.api.common.fluid.FluidSupportProvider;
import me.shedaniel.rei.api.common.plugins.REICommonPlugin;
import me.shedaniel.rei.api.common.registry.display.ServerDisplayRegistry;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.plugin.client.categories.crafting.filler.*;
import me.shedaniel.rei.plugin.client.displays.ClientsidedCookingDisplay;
import me.shedaniel.rei.plugin.client.displays.ClientsidedCraftingDisplay;
import me.shedaniel.rei.plugin.common.displays.*;
import me.shedaniel.rei.plugin.common.displays.anvil.DefaultAnvilDisplay;
import me.shedaniel.rei.plugin.common.displays.beacon.DefaultBeaconBaseDisplay;
import me.shedaniel.rei.plugin.common.displays.beacon.DefaultBeaconPaymentDisplay;
import me.shedaniel.rei.plugin.common.displays.brewing.DefaultBrewingDisplay;
import me.shedaniel.rei.plugin.common.displays.cooking.DefaultBlastingDisplay;
import me.shedaniel.rei.plugin.common.displays.cooking.DefaultSmeltingDisplay;
import me.shedaniel.rei.plugin.common.displays.cooking.DefaultSmokingDisplay;
import me.shedaniel.rei.plugin.common.displays.crafting.*;
import me.shedaniel.rei.plugin.common.displays.tag.TagNodes;
import net.minecraft.class_1755;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2960;
import net.minecraft.class_3611;
import net.minecraft.class_3859;
import net.minecraft.class_3861;
import net.minecraft.class_3862;
import net.minecraft.class_3920;
import net.minecraft.class_3955;
import net.minecraft.class_3956;
import net.minecraft.class_3975;
import net.minecraft.class_8060;
import net.minecraft.class_8062;
import net.minecraft.class_9304;
import net.minecraft.class_9334;
import net.minecraft.world.item.crafting.*;
import org.jetbrains.annotations.ApiStatus;

import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Stream;

@ApiStatus.Internal
public class DefaultPlugin implements BuiltinPlugin, REICommonPlugin {
    private static final CraftingRecipeFiller<?>[] CRAFTING_RECIPE_FILLERS = new CraftingRecipeFiller[]{
            new TippedArrowRecipeFiller(),
            new BannerDuplicateRecipeFiller(),
            new ShieldDecorationRecipeFiller(),
            new BookCloningRecipeFiller(),
            new FireworkRocketRecipeFiller(),
            new ArmorDyeRecipeFiller(),
            new MapCloningRecipeFiller(),
            new MapExtendingRecipeFiller()
    };
    
    static {
        TagNodes.init();
    }
    
    @Override
    public void registerItemComparators(ItemComparatorRegistry registry) {
        Function<class_1799, class_9304> enchantmentTag = stack -> {
            if (stack.method_57826(class_9334.field_49643)) {
                return stack.method_58694(class_9334.field_49643);
            }
            return stack.method_58694(class_9334.field_49633);
        };
        registry.register((context, stack) -> Objects.hashCode(enchantmentTag.apply(stack)), class_1802.field_8598);
        registry.registerComponents(class_1802.field_8574);
        registry.registerComponents(class_1802.field_8436);
        registry.registerComponents(class_1802.field_8150);
        registry.registerComponents(class_1802.field_8087);
        registry.register((context, stack) -> 0, class_1802.field_8639, class_1802.field_8204);
    }
    
    @Override
    public void registerFluidSupport(FluidSupportProvider support) {
        support.register(entry -> {
            class_1799 stack = entry.getValue();
            class_1792 item = stack.method_7909();
            if (item instanceof class_1755 bucketItem) {
                class_3611 fluid = FluidBucketHooks.getFluid(bucketItem);
                if (fluid != null) {
                    return CompoundEventResult.interruptTrue(Stream.of(EntryStacks.of(fluid, FluidStackHooks.bucketAmount())));
                }
            }
            return CompoundEventResult.pass();
        });
    }
    
    @Override
    public void registerDisplays(ServerDisplayRegistry registry) {
        registry.beginRecipeFiller(class_3955.class)
                .filterType(class_3956.field_17545)
                .fill(DefaultCraftingDisplay::of);
        registry.beginRecipeFiller(class_3861.class)
                .filterType(class_3956.field_17546)
                .fill(DefaultSmeltingDisplay::new);
        registry.beginRecipeFiller(class_3862.class)
                .filterType(class_3956.field_17548)
                .fill(DefaultSmokingDisplay::new);
        registry.beginRecipeFiller(class_3859.class)
                .filterType(class_3956.field_17547)
                .fill(DefaultBlastingDisplay::new);
        registry.beginRecipeFiller(class_3920.class)
                .filterType(class_3956.field_17549)
                .fill(DefaultCampfireDisplay::new);
        registry.beginRecipeFiller(class_3975.class)
                .filterType(class_3956.field_17641)
                .fill(DefaultStoneCuttingDisplay::new);
        registry.beginRecipeFiller(class_8060.class)
                .filterType(class_3956.field_25388)
                .fill(DefaultSmithingDisplay::ofTransforming);
        registry.beginRecipeFiller(class_8062.class)
                .filterType(class_3956.field_25388)
                .fillMultiple(DefaultSmithingDisplay::fromTrimming);
        
        for (CraftingRecipeFiller<?> filler : CRAFTING_RECIPE_FILLERS) {
            filler.registerDisplays(registry);
        }
    }
    
    @Override
    public void registerDisplaySerializer(DisplaySerializerRegistry registry) {
        registry.register(id("client/crafting/shaped"), ClientsidedCraftingDisplay.Shaped.SERIALIZER);
        registry.register(id("client/crafting/shapeless"), ClientsidedCraftingDisplay.Shapeless.SERIALIZER);
        registry.register(id("client/smelting"), ClientsidedCookingDisplay.Smelting.SERIALIZER);
        registry.register(id("client/smoking"), ClientsidedCookingDisplay.Smoking.SERIALIZER);
        registry.register(id("client/blasting"), ClientsidedCookingDisplay.Blasting.SERIALIZER);
        registry.register(id("default/crafting/shaped"), DefaultShapedDisplay.SERIALIZER);
        registry.register(id("default/crafting/shapeless"), DefaultShapelessDisplay.SERIALIZER);
        registry.register(id("default/crafting/custom"), DefaultCustomDisplay.SERIALIZER);
        registry.register(id("default/crafting/custom_shaped"), DefaultCustomShapedDisplay.SERIALIZER);
        registry.register(id("default/crafting/custom_shapeless"), DefaultCustomShapelessDisplay.SERIALIZER);
        registry.register(id("extension/crafting/map_extending"), MapExtendingCraftingDisplay.SERIALIZER);
        registry.register(id("default/smelting"), DefaultSmeltingDisplay.SERIALIZER);
        registry.register(id("default/smoking"), DefaultSmokingDisplay.SERIALIZER);
        registry.register(id("default/blasting"), DefaultBlastingDisplay.SERIALIZER);
        registry.register(id("default/campfire"), DefaultCampfireDisplay.SERIALIZER);
        registry.register(id("default/stone_cutting"), DefaultStoneCuttingDisplay.SERIALIZER);
        registry.register(id("default/stripping"), DefaultStrippingDisplay.SERIALIZER);
        registry.register(id("default/brewing"), DefaultBrewingDisplay.SERIALIZER);
        registry.register(id("default/composting"), DefaultCompostingDisplay.SERIALIZER);
        registry.register(id("default/fuel"), DefaultFuelDisplay.SERIALIZER);
        registry.register(id("default/smithing"), DefaultSmithingDisplay.SERIALIZER);
        registry.register(id("default/smithing/trimming"), DefaultSmithingDisplay.Trimming.SERIALIZER);
        registry.register(id("default/anvil"), DefaultAnvilDisplay.SERIALIZER);
        registry.register(id("default/beacon_base"), DefaultBeaconBaseDisplay.SERIALIZER);
        registry.register(id("default/beacon_payment"), DefaultBeaconPaymentDisplay.SERIALIZER);
        registry.register(id("default/tilling"), DefaultTillingDisplay.SERIALIZER);
        registry.register(id("default/pathing"), DefaultPathingDisplay.SERIALIZER);
        registry.register(id("default/waxing"), DefaultWaxingDisplay.SERIALIZER);
        registry.register(id("default/waxing_scraping"), DefaultWaxScrapingDisplay.SERIALIZER);
        registry.register(id("default/oxidizing"), DefaultOxidizingDisplay.SERIALIZER);
        registry.register(id("default/oxidizing_scraping"), DefaultOxidationScrapingDisplay.SERIALIZER);
        registry.register(id("roughlyenoughitems:default/information"), DefaultInformationDisplay.SERIALIZER);
    }
    
    @Override
    public double getPriority() {
        return -100;
    }
    
    private static class_2960 id(String path) {
        return class_2960.method_60654(path);
    }
}
