/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.common.displays.tag;

import com.google.common.collect.Maps;
import com.mojang.serialization.DataResult;
import dev.architectury.event.events.client.ClientLifecycleEvent;
import dev.architectury.networking.NetworkManager;
import dev.architectury.utils.Env;
import dev.architectury.utils.EnvExecutor;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import me.shedaniel.rei.api.common.util.UUIDUtils;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3222;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_7923;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;

@ApiStatus.Internal
public class TagNodes {
    public static final class_2960 REQUEST_TAGS_C2S_PACKET_ID = class_2960.method_60655("roughlyenoughitems", "request_tags_c2s");
    public static final class_2960 REQUEST_TAGS_S2C_PACKET_ID = class_2960.method_60655("roughlyenoughitems", "request_tags_s2c");
    
    public static final class_8710.class_9154<C2STagDataPacket> REQUEST_TAGS_C2S_PACKET_TYPE = new class_8710.class_9154<>(REQUEST_TAGS_C2S_PACKET_ID);
    public static final class_8710.class_9154<S2CTagDataPacket> REQUEST_TAGS_S2C_PACKET_TYPE = new class_8710.class_9154<>(REQUEST_TAGS_S2C_PACKET_ID);
    
    public static final Map<String, class_5321<? extends class_2378<?>>> TAG_DIR_MAP = new HashMap<>();
    public static final ThreadLocal<String> CURRENT_TAG_DIR = new ThreadLocal<>();
    public static final Map<String, Map<CollectionWrapper<?>, RawTagData>> RAW_TAG_DATA_MAP = new ConcurrentHashMap<>();
    public static final Map<class_5321<? extends class_2378<?>>, Map<class_2960, TagData>> TAG_DATA_MAP = new HashMap<>();
    public static Map<class_5321<? extends class_2378<?>>, Consumer<Consumer<DataResult<Map<class_2960, TagData>>>>> requestedTags = new HashMap<>();
    
    public static class CollectionWrapper<T> {
        private final Collection<T> collection;
        
        public CollectionWrapper(Collection<T> collection) {
            this.collection = collection;
        }
        
        @Override
        public boolean equals(Object obj) {
            return obj instanceof CollectionWrapper && ((CollectionWrapper) obj).collection == collection;
        }
        
        @Override
        public int hashCode() {
            return System.identityHashCode(collection);
        }
    }
    
    public record RawTagData(List<class_2960> otherElements, List<class_2960> otherTags) {
    }
    
    public record TagData(IntList otherElements, List<class_2960> otherTags) {
        public static final class_9139<class_9129, TagData> STREAM_CODEC = class_9139.method_56435(
                class_9135.method_56376(IntArrayList::new, class_9135.field_48550), TagData::otherElements,
                class_9135.method_56376(ArrayList::new, class_2960.field_48267), TagData::otherTags,
                TagData::new
        );
    }
    
    public record C2STagDataPacket(UUID uuid, class_2960 registryName) implements class_8710 {
        public static final class_9139<class_9129, C2STagDataPacket> STREAM_CODEC = class_9139.method_56435(
                UUIDUtils.STREAM_CODEC, C2STagDataPacket::uuid,
                class_2960.field_48267, C2STagDataPacket::registryName,
                C2STagDataPacket::new
        );
        
        @Override
        public @NotNull class_9154<? extends class_8710> method_56479() {
            return REQUEST_TAGS_C2S_PACKET_TYPE;
        }
    }
    
    public record S2CTagDataPacket(UUID uuid, Map<class_2960, TagData> map) implements class_8710 {
        public static final class_9139<class_9129, S2CTagDataPacket> STREAM_CODEC = class_9139.method_56435(
                UUIDUtils.STREAM_CODEC, S2CTagDataPacket::uuid,
                class_9135.method_56377(
                        Maps::newHashMapWithExpectedSize,
                        class_2960.field_48267,
                        TagData.STREAM_CODEC
                ), S2CTagDataPacket::map,
                S2CTagDataPacket::new
        );
        
        @Override
        public @NotNull class_9154<? extends class_8710> method_56479() {
            return REQUEST_TAGS_S2C_PACKET_TYPE;
        }
    }
    
    public static void init() {
        EnvExecutor.runInEnv(Env.CLIENT, () -> Client::init);
        EnvExecutor.runInEnv(Env.SERVER, () -> Server::init);
        
        NetworkManager.registerReceiver(
                NetworkManager.c2s(),
                REQUEST_TAGS_C2S_PACKET_TYPE,
                C2STagDataPacket.STREAM_CODEC,
                (C2STagDataPacket payload, NetworkManager.PacketContext context) -> {
                    class_5321<? extends class_2378<?>> registryKey = class_5321.method_29180(payload.registryName);
                    Map<class_2960, TagData> dataMap = TAG_DATA_MAP.getOrDefault(registryKey, Collections.emptyMap());
                    var packet = new S2CTagDataPacket(payload.uuid, dataMap);
                    NetworkManager.sendToPlayer((class_3222) context.getPlayer(), packet);
                }
        );
    }
    
    @Environment(EnvType.CLIENT)
    public static void requestTagData(class_5321<? extends class_2378<?>> resourceKey, Consumer<DataResult<Map<class_2960, TagData>>> callback) {
        if (class_310.method_1551().method_1576() != null) {
            callback.accept(DataResult.success(TAG_DATA_MAP.get(resourceKey)));
        } else if (!NetworkManager.canServerReceive(REQUEST_TAGS_C2S_PACKET_ID)) {
            callback.accept(DataResult.error(() -> "Cannot request tags from server"));
        } else if (requestedTags.containsKey(resourceKey)) {
            requestedTags.get(resourceKey).accept(callback);
            callback.accept(DataResult.success(TAG_DATA_MAP.getOrDefault(resourceKey, Collections.emptyMap())));
        } else {
            UUID uuid = UUID.randomUUID();
            var packet = new C2STagDataPacket(uuid, resourceKey.method_29177());
            Client.nextUUID = uuid;
            Client.nextResourceKey = resourceKey;
            List<Consumer<DataResult<Map<class_2960, TagData>>>> callbacks = new CopyOnWriteArrayList<>();
            callbacks.add(callback);
            Client.nextCallback = mapDataResult -> {
                requestedTags.put(resourceKey, c -> c.accept(mapDataResult));
                for (Consumer<DataResult<Map<class_2960, TagData>>> consumer : callbacks) {
                    consumer.accept(mapDataResult);
                }
            };
            requestedTags.put(resourceKey, callbacks::add);
            NetworkManager.sendToServer(packet);
        }
    }
    
    private static class Server {
        private static void init() {
            NetworkManager.registerS2CPayloadType(REQUEST_TAGS_S2C_PACKET_TYPE, S2CTagDataPacket.STREAM_CODEC);
        }
    }
    
    private static class Client {
        public static UUID nextUUID;
        public static class_5321<? extends class_2378<?>> nextResourceKey;
        public static Consumer<DataResult<Map<class_2960, TagData>>> nextCallback;
        
        private static void init() {
            ClientLifecycleEvent.CLIENT_LEVEL_LOAD.register(world -> {
                requestedTags.clear();
            });
            
            NetworkManager.registerReceiver(
                    NetworkManager.s2c(),
                    REQUEST_TAGS_S2C_PACKET_TYPE,
                    S2CTagDataPacket.STREAM_CODEC,
                    (S2CTagDataPacket payload, NetworkManager.PacketContext context) -> {
                        if (!nextUUID.equals(payload.uuid)) return;
                        
                        TAG_DATA_MAP.put(nextResourceKey, payload.map);
                        nextCallback.accept(DataResult.success(payload.map));
                        
                        nextUUID = null;
                        nextResourceKey = null;
                        nextCallback = null;
                    }
            );
        }
    }
    
    public static <T> void create(class_6862<T> tagKey, Consumer<DataResult<TagNode<T>>> callback) {
        class_2378<T> registry = ((class_2378<class_2378<T>>) class_7923.field_41167).method_31140((class_5321<class_2378<T>>) tagKey.comp_326());
        requestTagData(tagKey.comp_326(), result -> {
            callback.accept(result.flatMap(dataMap -> dataMap != null ? resolveTag(tagKey, registry, dataMap).orElse(DataResult.error(() -> "No tag data")) : DataResult.error(() -> "No tag data")));
        });
    }
    
    private static <T> Optional<DataResult<TagNode<T>>> resolveTag(class_6862<T> tagKey, class_2378<T> registry, Map<class_2960, TagData> tagDataMap) {
        TagData tagData = tagDataMap.get(tagKey.comp_327());
        if (tagData == null) return Optional.empty();
        
        TagNode<T> self = TagNode.ofReference(tagKey);
        List<class_6880<T>> holders = new ArrayList<>();
        for (int element : tagData.otherElements()) {
            Optional<class_6880.class_6883<T>> holder = registry.method_40265(element);
            if (holder.isPresent()) {
                holders.add(holder.get());
            }
        }
        if (!holders.isEmpty()) {
            self.addValuesChild(class_6885.method_40242(holders));
        }
        for (class_2960 childTagId : tagData.otherTags()) {
            class_6862<T> childTagKey = class_6862.method_40092(tagKey.comp_326(), childTagId);
            if (registry.method_46733(childTagKey).isPresent()) {
                Optional<DataResult<TagNode<T>>> resultOptional = resolveTag(childTagKey, registry, tagDataMap);
                if (resultOptional.isPresent()) {
                    DataResult<TagNode<T>> result = resultOptional.get();
                    if (result.error().isPresent())
                        return Optional.of(DataResult.error(() -> result.error().get().message()));
                    self.addChild(result.result().get());
                }
            }
        }
        return Optional.of(DataResult.success(self));
    }
}
