/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.common.displays;

import com.mojang.serialization.codecs.RecordCodecBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.display.DisplaySerializer;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import me.shedaniel.rei.plugin.common.BuiltinPlugin;
import me.shedaniel.rei.plugin.common.SmithingDisplay;
import net.minecraft.class_10711;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2960;
import net.minecraft.class_5455;
import net.minecraft.class_6880;
import net.minecraft.class_7225;
import net.minecraft.class_8053;
import net.minecraft.class_8054;
import net.minecraft.class_8055;
import net.minecraft.class_8056;
import net.minecraft.class_8060;
import net.minecraft.class_8062;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import net.minecraft.class_9334;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

public class DefaultSmithingDisplay extends BasicDisplay implements SmithingDisplay {
    public static final DisplaySerializer<DefaultSmithingDisplay> SERIALIZER = DisplaySerializer.of(
            RecordCodecBuilder.mapCodec(instance -> instance.group(
                    EntryIngredient.codec().listOf().fieldOf("inputs").forGetter(DefaultSmithingDisplay::getInputEntries),
                    EntryIngredient.codec().listOf().fieldOf("outputs").forGetter(DefaultSmithingDisplay::getOutputEntries),
                    SmithingRecipeType.CODEC.optionalFieldOf("type").forGetter(d -> d.type),
                    class_2960.field_25139.optionalFieldOf("location").forGetter(DefaultSmithingDisplay::getDisplayLocation)
            ).apply(instance, DefaultSmithingDisplay::new)),
            class_9139.method_56905(
                    EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                    DefaultSmithingDisplay::getInputEntries,
                    EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                    DefaultSmithingDisplay::getOutputEntries,
                    class_9135.method_56382(SmithingRecipeType.STREAM_CODEC),
                    d -> d.type,
                    class_9135.method_56382(class_2960.field_48267),
                    DefaultSmithingDisplay::getDisplayLocation,
                    DefaultSmithingDisplay::new
            ));
    
    protected final Optional<SmithingRecipeType> type;
    
    @ApiStatus.Experimental
    public static DefaultSmithingDisplay ofTransforming(class_8786<class_8060> recipe) {
        return new DefaultSmithingDisplay(
                List.of(
                        recipe.comp_1933().method_64722().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty()),
                        EntryIngredients.ofIngredient(recipe.comp_1933().method_64723()),
                        recipe.comp_1933().method_64724().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty())
                ),
                List.of(EntryIngredients.ofSlotDisplay(recipe.comp_1933().field_42033.method_66338())),
                Optional.of(SmithingRecipeType.TRANSFORM),
                Optional.of(recipe.comp_1932().method_29177())
        );
    }
    
    public static List<DefaultSmithingDisplay> fromTrimming(class_8786<class_8062> recipe) {
        class_5455 registryAccess = BasicDisplay.registryAccess();
        List<DefaultSmithingDisplay> displays = new ArrayList<>();
        class_6880<class_8056> trimPattern = recipe.comp_1933().field_56321;
        for (class_6880<class_1792> additionStack : (Iterable<class_6880<class_1792>>) recipe.comp_1933().method_64724().map(class_1856::method_8105).orElse(Stream.of())::iterator) {
            class_6880<class_8054> trimMaterial = getMaterialFromIngredient(registryAccess, additionStack)
                    .orElse(null);
            if (trimMaterial == null) continue;
            
            EntryIngredient baseIngredient = EntryIngredients.ofIngredient(recipe.comp_1933().method_64723());
            
            displays.add(new DefaultSmithingDisplay.Trimming(List.of(
                    recipe.comp_1933().method_64722().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty()),
                    baseIngredient,
                    EntryIngredients.ofItemHolder(additionStack)
            ), List.of(baseIngredient), Optional.of(SmithingRecipeType.TRIM), Optional.of(recipe.comp_1932().method_29177()), recipe.comp_1933().field_56321));
        }
        return displays;
    }
    
    public DefaultSmithingDisplay(List<EntryIngredient> inputs, List<EntryIngredient> outputs, Optional<class_2960> location) {
        this(inputs, outputs, Optional.empty(), location);
    }
    
    @ApiStatus.Experimental
    public DefaultSmithingDisplay(List<EntryIngredient> inputs, List<EntryIngredient> outputs, Optional<SmithingRecipeType> type, Optional<class_2960> location) {
        super(inputs, outputs, location);
        this.type = type;
    }
    
    @Override
    public CategoryIdentifier<?> getCategoryIdentifier() {
        return BuiltinPlugin.SMITHING;
    }
    
    @Override
    public DisplaySerializer<? extends Display> getSerializer() {
        return SERIALIZER;
    }
    
    @Nullable
    @Override
    public SmithingRecipeType type() {
        return type.orElse(null);
    }
    
    @ApiStatus.Experimental
    @ApiStatus.Internal
    public static EntryIngredient getTrimmingOutput(class_5455 registryAccess, class_6880<class_8056> trimPattern, EntryStack<?> base, EntryStack<?> addition) {
        if (base.getType() != VanillaEntryTypes.ITEM || addition.getType() != VanillaEntryTypes.ITEM) return EntryIngredient.empty();
        class_1799 baseItem = base.castValue();
        class_1799 additionItem = addition.castValue();
        if (trimPattern == null) return EntryIngredient.empty();
        class_6880<class_8054> trimMaterial = class_8055.method_48440(registryAccess, additionItem)
                .orElse(null);
        if (trimMaterial == null) return EntryIngredient.empty();
        class_8053 armorTrim = new class_8053(trimMaterial, trimPattern);
        class_8053 trim = baseItem.method_58694(class_9334.field_49607);
        if (Objects.equals(trim, armorTrim)) return EntryIngredient.empty();
        class_1799 newItem = baseItem.method_46651(1);
        newItem.method_57379(class_9334.field_49607, armorTrim);
        return EntryIngredients.of(newItem);
    }
    
    private static Optional<class_6880<class_8054>> getMaterialFromIngredient(class_7225.class_7874 provider, class_6880<class_1792> item) {
        class_10711 providesTrimMaterial = new class_1799(item).method_58694(class_9334.field_56397);
        return providesTrimMaterial != null ? providesTrimMaterial.method_67212(provider) : Optional.empty();
    }
    
    public static class Trimming extends DefaultSmithingDisplay implements SmithingDisplay.Trimming {
        public static final DisplaySerializer<DefaultSmithingDisplay.Trimming> SERIALIZER = DisplaySerializer.of(
                RecordCodecBuilder.mapCodec(instance -> instance.group(
                        EntryIngredient.codec().listOf().fieldOf("inputs").forGetter(DefaultSmithingDisplay.Trimming::getInputEntries),
                        EntryIngredient.codec().listOf().fieldOf("outputs").forGetter(DefaultSmithingDisplay.Trimming::getOutputEntries),
                        SmithingRecipeType.CODEC.optionalFieldOf("smithing_type").forGetter(d -> d.type),
                        class_2960.field_25139.optionalFieldOf("location").forGetter(DefaultSmithingDisplay.Trimming::getDisplayLocation),
                        class_8056.field_42015.fieldOf("pattern").forGetter(DefaultSmithingDisplay.Trimming::pattern)
                ).apply(instance, DefaultSmithingDisplay.Trimming::new)),
                class_9139.method_56906(
                        EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                        DefaultSmithingDisplay.Trimming::getInputEntries,
                        EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                        DefaultSmithingDisplay.Trimming::getOutputEntries,
                        class_9135.method_56382(SmithingRecipeType.STREAM_CODEC),
                        d -> d.type,
                        class_9135.method_56382(class_2960.field_48267),
                        DefaultSmithingDisplay.Trimming::getDisplayLocation,
                        class_8056.field_49283,
                        DefaultSmithingDisplay.Trimming::pattern,
                        DefaultSmithingDisplay.Trimming::new
                ));
        
        private final class_6880<class_8056> pattern;
        
        public Trimming(List<EntryIngredient> inputs, List<EntryIngredient> outputs, Optional<SmithingRecipeType> type, Optional<class_2960> location, class_6880<class_8056> pattern) {
            super(inputs, outputs, type, location);
            this.pattern = pattern;
        }
        
        @Override
        public class_6880<class_8056> pattern() {
            return pattern;
        }
        
        @Override
        public DisplaySerializer<? extends Display> getSerializer() {
            return DefaultSmithingDisplay.Trimming.SERIALIZER;
        }
    }
}
