/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.api.client.registry.transfer;

import me.shedaniel.math.Point;
import me.shedaniel.rei.api.client.gui.widgets.Tooltip;
import me.shedaniel.rei.api.client.registry.transfer.simple.SimpleTransferHandler;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.impl.ClientInternals;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1703;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_465;
import net.minecraft.class_5632;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

/**
 * Handler for display transfer, only executed on the client, implementations of this handler should sync recipes to the server to transfer recipes themselves.
 * <p>
 * REI provides a {@link SimpleTransferHandler} that handles displays,
 * as a lightweight and simple way to implement recipe transfers.
 *
 * @see TransferHandlerRegistry
 * @see me.shedaniel.rei.api.client.registry.transfer.simple.SimpleTransferHandler
 */
@Environment(EnvType.CLIENT)
public interface TransferHandler extends Comparable<TransferHandler> {
    /**
     * @return the priority of this handler, higher priorities will be called first.
     */
    default double getPriority() {
        return 0d;
    }
    
    /**
     * Returns whether the transfer handler is applicable to the current context.
     *
     * @param context the context of the transfer
     * @return whether the transfer handler is applicable to the current context.
     */
    default ApplicabilityResult checkApplicable(Context context) {
        return ApplicabilityResult.createApplicable();
    }
    
    /**
     * Handles the transfer of the specified display.
     * <p>
     * If {@link Context#isActuallyCrafting()} returns {@code false},
     * the transfer handler should <b>not</b> instantiate the transfer.
     *
     * @param context the context
     * @return the result of the transfer
     */
    Result handle(Context context);
    
    /**
     * {@inheritDoc}
     */
    @Override
    default int compareTo(TransferHandler o) {
        return Double.compare(getPriority(), o.getPriority());
    }
    
    @ApiStatus.NonExtendable
    interface Result {
        /**
         * Creates a successful result, no further handlers will be called.
         */
        static Result createSuccessful() {
            return new ResultImpl().color(0x00000000);
        }
        
        /**
         * Creates a passing result, further handlers will be called.
         * This will also mark the handler as not applicable.
         */
        static Result createNotApplicable() {
            return new ResultImpl(false).color(0x00000000);
        }
        
        /**
         * Creates a passing result, further handlers will be called.
         *
         * @param error The error itself
         */
        static Result createFailed(class_2561 error) {
            return new ResultImpl(error, 0x67ff0000);
        }
        
        /**
         * Creates a passing result, further handlers will be called.
         * The special color will be applied if this is the last handler.
         *
         * @param error The error itself
         * @param color A special color for the button
         */
        static Result createFailedCustomButtonColor(class_2561 error, int color) {
            return createFailed(error).color(color);
        }
        
        /**
         * Forces this handler to be the last handler, no further handlers will be called.
         */
        default Result blocksFurtherHandling() {
            return blocksFurtherHandling(true);
        }
        
        /**
         * Forces this handler to be the last handler, no further handlers will be called.
         */
        Result blocksFurtherHandling(boolean returnsToScreen);
        
        /**
         * @return the color in which the button should be displayed in.
         */
        int getColor();
        
        /**
         * Sets the color in which the button should be displayed in.
         */
        Result color(int color);
        
        /**
         * Sets the renderer of the transfer, to be used when the mouse hovers the plus button.
         */
        Result renderer(TransferHandlerRenderer renderer);
        
        /**
         * Overrides the tooltip renderer completely.
         *
         * @param renderer the renderer
         * @return the result
         */
        @ApiStatus.Experimental
        Result overrideTooltipRenderer(BiConsumer<Point, TooltipSink> renderer);
        
        /**
         * Adds a line of tooltip to the result.
         *
         * @param component the component to add
         * @return the result
         */
        Result tooltip(class_2561 component);
        
        /**
         * Adds a line of tooltip to the result.
         *
         * @param component the component to add
         * @return the result
         * @since 8.3
         */
        @ApiStatus.Experimental
        Result tooltip(class_5632 component);
        
        /**
         * Adds a tooltip component for the missing items.
         *
         * @param stacks the missing stacks
         * @return the result
         * @since 8.3
         */
        @ApiStatus.Experimental
        Result tooltipMissing(List<EntryIngredient> stacks);
        
        /**
         * @return whether this handler has successfully handled the transfer.
         */
        boolean isSuccessful();
        
        /**
         * @return whether this handler should be the last handler.
         */
        boolean isBlocking();
        
        /**
         * Applicable if {@link #isSuccessful()} is true.
         *
         * @return whether to return to the previous screen rather than staying open
         */
        boolean isReturningToScreen();
        
        /**
         * @return whether the handler is applicable.
         */
        boolean isApplicable();
        
        /**
         * Applicable if {@link #isSuccessful()} is false.
         *
         * @return the error message
         */
        class_2561 getError();
        
        @Environment(EnvType.CLIENT)
        @ApiStatus.Internal
        TransferHandlerRenderer getRenderer(TransferHandler handler, Context context);
        
        @Environment(EnvType.CLIENT)
        @ApiStatus.Internal
        BiConsumer<Point, TooltipSink> getTooltipRenderer();
        
        @ApiStatus.Internal
        void fillTooltip(List<Tooltip.Entry> entries);
        
        @FunctionalInterface
        interface TooltipSink {
            void accept(Tooltip tooltip);
        }
    }
    
    @ApiStatus.NonExtendable
    interface Context {
        static Context create(boolean actuallyCrafting, boolean stackedCrafting, @Nullable class_465<?> containerScreen, Display display) {
            return new ContextImpl(actuallyCrafting, stackedCrafting, containerScreen, () -> display);
        }
        
        default class_310 getMinecraft() {
            return class_310.method_1551();
        }
        
        /**
         * Returns whether we should actually move the items.
         *
         * @return whether we should actually move the items.
         */
        boolean isActuallyCrafting();
        
        boolean isStackedCrafting();
        
        Display getDisplay();
        
        @Nullable
        class_465<?> getContainerScreen();
        
        @Nullable
        default class_1703 getMenu() {
            return getContainerScreen() == null ? null : getContainerScreen().method_17577();
        }
    }
    
    interface ApplicabilityResult {
        static ApplicabilityResult createNotApplicable() {
            return new ApplicabilityResultImpl(false, null);
        }
        
        static ApplicabilityResult createApplicableWithError(class_2561 error) {
            return createApplicableWithError(Result.createFailed(error));
        }
        
        static ApplicabilityResult createApplicableWithError(TransferHandler.Result result) {
            if (result == null) throw new NullPointerException("result");
            if (!result.isApplicable()) throw new IllegalArgumentException("result is not applicable");
            return new ApplicabilityResultImpl(true, result);
        }
        
        static ApplicabilityResult createApplicable() {
            return new ApplicabilityResultImpl(true, null);
        }
        
        boolean isApplicable();
        
        boolean isSuccessful();
        
        @Nullable
        TransferHandler.Result getError();
    }
    
    @ApiStatus.Internal
    final class ResultImpl implements Result {
        private boolean successful, applicable, returningToScreen, blocking;
        private class_2561 error;
        private List<Tooltip.Entry> tooltips = new ArrayList<>();
        private TransferHandlerRenderer renderer;
        private BiConsumer<Point, TooltipSink> tooltipRenderer;
        private int color;
        
        private ResultImpl() {
            this(true, true);
        }
        
        public ResultImpl(boolean applicable) {
            this(false, applicable);
        }
        
        public ResultImpl(boolean successful, boolean applicable) {
            this.successful = successful;
            this.applicable = applicable;
        }
        
        public ResultImpl(class_2561 error, int color) {
            this.successful = false;
            this.applicable = true;
            this.error = error;
            this.color = color;
        }
        
        @Override
        public Result blocksFurtherHandling(boolean returningToScreen) {
            this.blocking = true;
            this.returningToScreen = returningToScreen;
            return this;
        }
        
        @Override
        public int getColor() {
            return color;
        }
        
        @Override
        public TransferHandler.Result color(int color) {
            this.color = color;
            return this;
        }
        
        @Override
        public Result renderer(TransferHandlerRenderer renderer) {
            this.renderer = renderer;
            return this;
        }
        
        @Override
        public Result overrideTooltipRenderer(BiConsumer<Point, TooltipSink> renderer) {
            this.tooltipRenderer = renderer;
            return this;
        }
        
        @Override
        public Result tooltip(class_2561 component) {
            this.tooltips.add(Tooltip.entry(component));
            return this;
        }
        
        @Override
        public Result tooltip(class_5632 component) {
            this.tooltips.add(Tooltip.entry(component));
            return this;
        }
        
        @Override
        public Result tooltipMissing(List<EntryIngredient> ingredients) {
            return tooltip(ClientInternals.createMissingTooltip(ingredients));
        }
        
        @Override
        public boolean isSuccessful() {
            return successful;
        }
        
        @Override
        public boolean isBlocking() {
            return successful || blocking;
        }
        
        @Override
        public boolean isApplicable() {
            return applicable;
        }
        
        @Override
        public boolean isReturningToScreen() {
            return returningToScreen;
        }
        
        @Override
        public class_2561 getError() {
            return error;
        }
        
        @Override
        public TransferHandlerRenderer getRenderer(TransferHandler handler, Context context) {
            return renderer;
        }
        
        @Override
        public BiConsumer<Point, TooltipSink> getTooltipRenderer() {
            return tooltipRenderer;
        }
        
        @Override
        public void fillTooltip(List<Tooltip.Entry> entries) {
            if (isApplicable()) {
                if (!isSuccessful()) {
                    entries.add(Tooltip.entry(getError()));
                }
                entries.addAll(tooltips);
            }
        }
    }
    
    @ApiStatus.Internal
    final class ContextImpl implements Context {
        private boolean actuallyCrafting;
        private boolean stackedCrafting;
        private class_465<?> containerScreen;
        private Supplier<Display> recipeDisplaySupplier;
        
        private ContextImpl(boolean actuallyCrafting, boolean stackedCrafting, class_465<?> containerScreen, Supplier<Display> recipeDisplaySupplier) {
            this.actuallyCrafting = actuallyCrafting;
            this.stackedCrafting = stackedCrafting;
            this.containerScreen = containerScreen;
            this.recipeDisplaySupplier = recipeDisplaySupplier;
        }
        
        @Override
        public boolean isActuallyCrafting() {
            return actuallyCrafting;
        }
        
        @Override
        public boolean isStackedCrafting() {
            return stackedCrafting;
        }
        
        @Override
        public class_465<?> getContainerScreen() {
            return containerScreen;
        }
        
        @Override
        public Display getDisplay() {
            return recipeDisplaySupplier.get();
        }
    }
    
    @ApiStatus.Internal
    final class ApplicabilityResultImpl implements ApplicabilityResult {
        private boolean applicable;
        @Nullable
        private Result error;
        
        public ApplicabilityResultImpl(boolean applicable, @Nullable Result error) {
            this.applicable = applicable;
            this.error = error;
        }
        
        @Override
        public boolean isApplicable() {
            return this.applicable;
        }
        
        @Override
        public boolean isSuccessful() {
            return this.isApplicable() && this.error == null;
        }
        
        @Nullable
        @Override
        public TransferHandler.Result getError() {
            return this.error;
        }
    }
}
