/*
 * This file is part of fabric-loom, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021-2025 FabricMC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.fabricmc.loom.extension;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;

import org.gradle.api.Action;
import org.gradle.api.NamedDomainObjectContainer;
import org.gradle.api.NamedDomainObjectList;
import org.gradle.api.Project;
import org.gradle.api.artifacts.Dependency;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.FileCollection;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.model.ObjectFactory;
import org.gradle.api.provider.ListProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;
import org.gradle.api.provider.SetProperty;
import org.gradle.api.publish.maven.MavenPublication;
import org.gradle.api.tasks.SourceSet;

import net.fabricmc.loom.LoomGradleExtension;
import net.fabricmc.loom.api.ForgeExtensionAPI;
import net.fabricmc.loom.api.InterfaceInjectionExtensionAPI;
import net.fabricmc.loom.api.LoomGradleExtensionAPI;
import net.fabricmc.loom.api.MixinExtensionAPI;
import net.fabricmc.loom.api.ModSettings;
import net.fabricmc.loom.api.NeoForgeExtensionAPI;
import net.fabricmc.loom.api.RemapConfigurationSettings;
import net.fabricmc.loom.api.decompilers.DecompilerOptions;
import net.fabricmc.loom.api.mappings.intermediate.IntermediateMappingsProvider;
import net.fabricmc.loom.api.mappings.layered.MappingsNamespace;
import net.fabricmc.loom.api.mappings.layered.spec.LayeredMappingSpecBuilder;
import net.fabricmc.loom.api.processor.MinecraftJarProcessor;
import net.fabricmc.loom.api.remapping.RemapperExtension;
import net.fabricmc.loom.api.remapping.RemapperParameters;
import net.fabricmc.loom.configuration.RemapConfigurations;
import net.fabricmc.loom.configuration.ide.RunConfig;
import net.fabricmc.loom.configuration.ide.RunConfigSettings;
import net.fabricmc.loom.configuration.processors.JarProcessor;
import net.fabricmc.loom.configuration.providers.mappings.LayeredMappingSpec;
import net.fabricmc.loom.configuration.providers.mappings.LayeredMappingSpecBuilderImpl;
import net.fabricmc.loom.configuration.providers.mappings.LayeredMappingsFactory;
import net.fabricmc.loom.configuration.providers.minecraft.ManifestLocations;
import net.fabricmc.loom.configuration.providers.minecraft.MinecraftJarConfiguration;
import net.fabricmc.loom.configuration.providers.minecraft.MinecraftMetadataProvider;
import net.fabricmc.loom.configuration.providers.minecraft.MinecraftSourceSets;
import net.fabricmc.loom.task.GenerateSourcesTask;
import net.fabricmc.loom.util.DeprecationHelper;
import net.fabricmc.loom.util.Lazy;
import net.fabricmc.loom.util.MirrorUtil;
import net.fabricmc.loom.util.ModPlatform;
import net.fabricmc.loom.util.fmj.FabricModJson;
import net.fabricmc.loom.util.fmj.FabricModJsonHelpers;
import net.fabricmc.loom.util.gradle.GradleUtils;
import net.fabricmc.loom.util.gradle.SourceSetHelper;

/**
 * This class implements the public extension api.
 */
public abstract class LoomGradleExtensionApiImpl implements LoomGradleExtensionAPI {
	private static final String FORGE_PROPERTY = "loom.forge";
	private static final String PLATFORM_PROPERTY = "loom.platform";

	protected final DeprecationHelper deprecationHelper;
	@Deprecated()
	protected final ListProperty<JarProcessor> jarProcessors;
	protected final ConfigurableFileCollection log4jConfigs;
	protected final RegularFileProperty accessWidener;
	protected final RegularFileProperty fabricModJsonPath;
	protected final ManifestLocations versionsManifests;
	protected final Property<String> customMetadata;
	protected final SetProperty<String> knownIndyBsms;
	protected final Property<Boolean> transitiveAccessWideners;
	protected final Property<Boolean> modProvidedJavadoc;
	protected final Property<String> intermediary;
	protected final Property<IntermediateMappingsProvider> intermediateMappingsProvider;
	private final Property<Boolean> runtimeOnlyLog4j;
	private final Property<Boolean> splitModDependencies;
	private final Property<MinecraftJarConfiguration<?, ?, ?>> minecraftJarConfiguration;
	private final Property<Boolean> splitEnvironmentalSourceSet;
	private final InterfaceInjectionExtensionAPI interfaceInjectionExtension;

	private final NamedDomainObjectContainer<RunConfigSettings> runConfigs;
	private final NamedDomainObjectContainer<DecompilerOptions> decompilers;
	private final NamedDomainObjectContainer<ModSettings> mods;
	private final NamedDomainObjectList<RemapConfigurationSettings> remapConfigurations;
	private final ListProperty<MinecraftJarProcessor<?>> minecraftJarProcessors;
	protected final ListProperty<RemapperExtensionHolder> remapperExtensions;

	// A common mistake with layered mappings is to call the wrong `officialMojangMappings` method, use this to keep track of when we are building a layered mapping spec.
	protected final ThreadLocal<Boolean> layeredSpecBuilderScope = ThreadLocal.withInitial(() -> false);
	public static final String DEFAULT_INTERMEDIARY_URL = "https://maven.fabricmc.net/net/fabricmc/intermediary/%1$s/intermediary-%1$s-v2.jar";

	protected boolean hasEvaluatedLayeredMappings = false;
	protected final Map<LayeredMappingSpec, LayeredMappingsFactory> layeredMappingsDependencyMap = new HashMap<>();

	// ===================
	//  Architectury Loom
	// ===================
	private Provider<ModPlatform> platform;
	private final Property<Boolean> silentMojangMappingsLicense;
	public Boolean generateSrgTiny = null;
	private final List<String> tasksBeforeRun = Collections.synchronizedList(new ArrayList<>());
	public final List<Consumer<RunConfig>> settingsPostEdit = new ArrayList<>();

	protected LoomGradleExtensionApiImpl(Project project, LoomFiles directories) {
		this.jarProcessors = project.getObjects().listProperty(JarProcessor.class)
				.empty();
		this.log4jConfigs = project.files(directories.getDefaultLog4jConfigFile());
		this.accessWidener = project.getObjects().fileProperty();
		this.fabricModJsonPath = project.getObjects().fileProperty();
		this.versionsManifests = new ManifestLocations();
		this.versionsManifests.add("mojang", MirrorUtil.getVersionManifests(project), -2);
		this.versionsManifests.add("fabric_experimental", MirrorUtil.getExperimentalVersions(project), -1);
		this.customMetadata = project.getObjects().property(String.class);
		this.knownIndyBsms = project.getObjects().setProperty(String.class).convention(Set.of(
				"java/lang/invoke/StringConcatFactory",
				"java/lang/runtime/ObjectMethods",
				"org/codehaus/groovy/vmplugin/v8/IndyInterface"
		));
		this.knownIndyBsms.finalizeValueOnRead();
		this.transitiveAccessWideners = project.getObjects().property(Boolean.class)
				.convention(true);
		this.transitiveAccessWideners.finalizeValueOnRead();
		this.modProvidedJavadoc = project.getObjects().property(Boolean.class)
				.convention(project.provider(() -> !isForgeLike()));
		this.modProvidedJavadoc.finalizeValueOnRead();
		this.intermediary = project.getObjects().property(String.class)
				.convention(DEFAULT_INTERMEDIARY_URL);

		this.intermediateMappingsProvider = project.getObjects().property(IntermediateMappingsProvider.class);
		this.intermediateMappingsProvider.finalizeValueOnRead();

		this.deprecationHelper = new DeprecationHelper.ProjectBased(project);

		this.runConfigs = project.container(RunConfigSettings.class,
				baseName -> project.getObjects().newInstance(RunConfigSettings.class, project, baseName));
		this.decompilers = project.getObjects().domainObjectContainer(DecompilerOptions.class);
		this.mods = project.getObjects().domainObjectContainer(ModSettings.class);
		this.remapConfigurations = project.getObjects().namedDomainObjectList(RemapConfigurationSettings.class);
		//noinspection unchecked
		this.minecraftJarProcessors = (ListProperty<MinecraftJarProcessor<?>>) (Object) project.getObjects().listProperty(MinecraftJarProcessor.class);
		this.minecraftJarProcessors.finalizeValueOnRead();

		//noinspection unchecked
		this.minecraftJarConfiguration = project.getObjects().property((Class<MinecraftJarConfiguration<?, ?, ?>>) (Class<?>) MinecraftJarConfiguration.class)
				.convention(project.provider(() -> {
					final LoomGradleExtension extension = LoomGradleExtension.get(project);
					final MinecraftMetadataProvider metadataProvider = extension.getMetadataProvider();

					// if no configuration is selected by the user, attempt to select one
					// based on the mc version and which sides are present for it
					if (!metadataProvider.getVersionMeta().hasServer()) {
						return MinecraftJarConfiguration.CLIENT_ONLY;
					} else if (!metadataProvider.getVersionMeta().hasClient()) {
						return MinecraftJarConfiguration.SERVER_ONLY;
					} else if (!metadataProvider.getVersionMeta().isLegacyVersion()) {
						return MinecraftJarConfiguration.MERGED;
					} else {
						return MinecraftJarConfiguration.LEGACY_MERGED;
					}
				}));
		this.minecraftJarConfiguration.finalizeValueOnRead();

		this.accessWidener.finalizeValueOnRead();
		this.getGameJarProcessors().finalizeValueOnRead();

		this.runtimeOnlyLog4j = project.getObjects().property(Boolean.class).convention(false);
		this.runtimeOnlyLog4j.finalizeValueOnRead();

		this.splitModDependencies = project.getObjects().property(Boolean.class).convention(true);
		this.splitModDependencies.finalizeValueOnRead();

		this.interfaceInjectionExtension = project.getObjects().newInstance(InterfaceInjectionExtensionAPI.class);
		this.interfaceInjectionExtension.getIsEnabled().convention(true);

		this.splitEnvironmentalSourceSet = project.getObjects().property(Boolean.class).convention(false);
		this.splitEnvironmentalSourceSet.finalizeValueOnRead();

		remapperExtensions = project.getObjects().listProperty(RemapperExtensionHolder.class);
		remapperExtensions.finalizeValueOnRead();

		// Enable dep iface injection by default
		interfaceInjection(interfaceInjection -> {
			interfaceInjection.getEnableDependencyInterfaceInjection().convention(true).finalizeValueOnRead();
		});
		this.platform = project.provider(Lazy.of(() -> {
			Object platformProperty = GradleUtils.getProperty(project, PLATFORM_PROPERTY);

			if (platformProperty != null) {
				ModPlatform platform = ModPlatform.valueOf(Objects.toString(platformProperty).toUpperCase(Locale.ROOT));

				if (platform.isExperimental()) {
					project.getLogger().lifecycle("{} support is experimental. Please report any issues!", platform.displayName());
				}

				return platform;
			}

			Object forgeProperty = GradleUtils.getProperty(project, FORGE_PROPERTY);

			if (forgeProperty != null) {
				project.getLogger().warn("Project " + project.getPath() + " is using property " + FORGE_PROPERTY + " to enable forge mode. Please use '" + PLATFORM_PROPERTY + " = forge' instead!");
				return Boolean.parseBoolean(Objects.toString(forgeProperty)) ? ModPlatform.FORGE : ModPlatform.FABRIC;
			}

			return ModPlatform.FABRIC;
		})::get);
		this.silentMojangMappingsLicense = project.getObjects().property(Boolean.class).convention(false);
		this.silentMojangMappingsLicense.finalizeValueOnRead();
	}

	@Override
	public DeprecationHelper getDeprecationHelper() {
		return deprecationHelper;
	}

	@Override
	public RegularFileProperty getAccessWidenerPath() {
		return accessWidener;
	}

	@Override
	public RegularFileProperty getFabricModJsonPath() {
		return fabricModJsonPath;
	}

	@Override
	public NamedDomainObjectContainer<DecompilerOptions> getDecompilerOptions() {
		return decompilers;
	}

	@Override
	public void decompilers(Action<NamedDomainObjectContainer<DecompilerOptions>> action) {
		action.execute(decompilers);
	}

	@Override
	public ListProperty<JarProcessor> getGameJarProcessors() {
		return jarProcessors;
	}

	@Override
	public ListProperty<MinecraftJarProcessor<?>> getMinecraftJarProcessors() {
		return minecraftJarProcessors;
	}

	@Override
	public void addMinecraftJarProcessor(Class<? extends MinecraftJarProcessor<?>> clazz, Object... parameters) {
		getMinecraftJarProcessors().add(getProject().getObjects().newInstance(clazz, parameters));
	}

	@Override
	public Dependency officialMojangMappings() {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot use Mojang mappings in a non-obfuscated environment");
		}

		if (layeredSpecBuilderScope.get()) {
			throw new IllegalStateException("Use `officialMojangMappings()` when configuring layered mappings, not the extension method `loom.officialMojangMappings()`");
		}

		return layered(LayeredMappingSpecBuilder::officialMojangMappings);
	}

	@Override
	public Dependency layered(Action<LayeredMappingSpecBuilder> action) {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot configure layered mappings in a non-obfuscated environment");
		}

		if (hasEvaluatedLayeredMappings) {
			throw new IllegalStateException("Layered mappings have already been evaluated");
		}

		LayeredMappingSpecBuilderImpl builder = new LayeredMappingSpecBuilderImpl(this);

		layeredSpecBuilderScope.set(true);
		action.execute(builder);
		layeredSpecBuilderScope.set(false);

		final LayeredMappingSpec builtSpec = builder.build();
		final LayeredMappingsFactory layeredMappingsFactory = layeredMappingsDependencyMap.computeIfAbsent(builtSpec, LayeredMappingsFactory::new);
		return layeredMappingsFactory.createDependency(getProject());
	}

	@Override
	public void runs(Action<NamedDomainObjectContainer<RunConfigSettings>> action) {
		action.execute(runConfigs);
	}

	@Override
	public NamedDomainObjectContainer<RunConfigSettings> getRunConfigs() {
		return runConfigs;
	}

	@Override
	public ConfigurableFileCollection getLog4jConfigs() {
		return log4jConfigs;
	}

	@Override
	public void mixin(Action<MixinExtensionAPI> action) {
		action.execute(getMixin());
	}

	@Override
	public ManifestLocations getVersionsManifests() {
		return versionsManifests;
	}

	@Override
	public Property<String> getCustomMinecraftMetadata() {
		return customMetadata;
	}

	@Override
	public SetProperty<String> getKnownIndyBsms() {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot configure known indyBsms in a non-obfuscated environment");
		}

		return knownIndyBsms;
	}

	@Override
	public String getModVersion() {
		List<FabricModJson> fabricModJsons = FabricModJsonHelpers.getModsInProject(getProject());

		if (fabricModJsons.isEmpty()) {
			throw new RuntimeException("Could not find a fabric.mod.json file in the main sourceset");
		}

		return fabricModJsons.getFirst().getModVersion();
	}

	@Override
	public Property<Boolean> getEnableTransitiveAccessWideners() {
		return transitiveAccessWideners;
	}

	@Override
	public Property<Boolean> getEnableModProvidedJavadoc() {
		return modProvidedJavadoc;
	}

	protected abstract Project getProject();

	protected abstract LoomFiles getFiles();

	@Override
	public Property<String> getIntermediaryUrl() {
		return intermediary;
	}

	@Override
	public IntermediateMappingsProvider getIntermediateMappingsProvider() {
		if (LoomGradleExtension.get(getProject()).disableObfuscation()) {
			throw new UnsupportedOperationException("Cannot get intermediate mappings provider in a non-obfuscated environment");
		}

		return intermediateMappingsProvider.get();
	}

	@Override
	public void setIntermediateMappingsProvider(IntermediateMappingsProvider intermediateMappingsProvider) {
		this.intermediateMappingsProvider.set(intermediateMappingsProvider);
	}

	@Override
	public <T extends IntermediateMappingsProvider> void setIntermediateMappingsProvider(Class<T> clazz, Action<T> action) {
		T provider = getProject().getObjects().newInstance(clazz);
		configureIntermediateMappingsProviderInternal(provider);
		action.execute(provider);
		setIntermediateMappingsProvider(provider);
	}

	@Override
	public File getMappingsFile() {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot get mappings file in a non-obfuscated environment");
		}

		return LoomGradleExtension.get(getProject()).getMappingConfiguration().tinyMappings.toFile();
	}

	@Override
	public GenerateSourcesTask getDecompileTask(DecompilerOptions options, boolean client) {
		final String decompilerName = options.getFormattedName();
		final String taskName;

		if (areEnvironmentSourceSetsSplit()) {
			taskName = "gen%sSourcesWith%s".formatted(client ? "ClientOnly" : "Common", decompilerName);
		} else {
			taskName = "genSourcesWith" + decompilerName;
		}

		return (GenerateSourcesTask) getProject().getTasks().getByName(taskName);
	}

	protected abstract <T extends IntermediateMappingsProvider> void configureIntermediateMappingsProviderInternal(T provider);

	@Override
	public void disableDeprecatedPomGeneration(MavenPublication publication) {
		net.fabricmc.loom.configuration.MavenPublication.excludePublication(publication);
	}

	@Override
	public Property<MinecraftJarConfiguration<?, ?, ?>> getMinecraftJarConfiguration() {
		return minecraftJarConfiguration;
	}

	@Override
	public Property<Boolean> getRuntimeOnlyLog4j() {
		return runtimeOnlyLog4j;
	}

	@Override
	public Property<Boolean> getSplitModDependencies() {
		return splitModDependencies;
	}

	@Override
	public void splitEnvironmentSourceSets() {
		splitMinecraftJar();

		splitEnvironmentalSourceSet.set(true);

		// We need to lock these values, as we setup the new source sets right away.
		splitEnvironmentalSourceSet.finalizeValue();
		minecraftJarConfiguration.finalizeValue();

		MinecraftSourceSets.get(getProject()).evaluateSplit(getProject());
	}

	@Override
	public boolean areEnvironmentSourceSetsSplit() {
		return splitEnvironmentalSourceSet.get();
	}

	@Override
	public InterfaceInjectionExtensionAPI getInterfaceInjection() {
		return interfaceInjectionExtension;
	}

	@Override
	public void mods(Action<NamedDomainObjectContainer<ModSettings>> action) {
		action.execute(getMods());
	}

	@Override
	public NamedDomainObjectContainer<ModSettings> getMods() {
		return mods;
	}

	@Override
	public NamedDomainObjectList<RemapConfigurationSettings> getRemapConfigurations() {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot get remap configurations in a non-obfuscated environment");
		}

		return remapConfigurations;
	}

	@Override
	public RemapConfigurationSettings addRemapConfiguration(String name, Action<RemapConfigurationSettings> action) {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot add remap configuration in a non-obfuscated environment");
		}

		final RemapConfigurationSettings configurationSettings = getProject().getObjects().newInstance(RemapConfigurationSettings.class, name);

		// TODO remove in 2.0, this is a fallback to mimic the previous (Broken) behaviour
		configurationSettings.getSourceSet().convention(SourceSetHelper.getMainSourceSet(getProject()));

		action.execute(configurationSettings);
		RemapConfigurations.applyToProject(getProject(), configurationSettings);
		remapConfigurations.add(configurationSettings);

		return configurationSettings;
	}

	@Override
	public void createRemapConfigurations(SourceSet sourceSet) {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot create remap configurations in a non-obfuscated environment");
		}

		RemapConfigurations.setupForSourceSet(getProject(), sourceSet);
	}

	@Override
	public <T extends RemapperParameters> void addRemapperExtension(Class<? extends RemapperExtension<T>> remapperExtensionClass, Class<T> parametersClass, Action<T> parameterAction) {
		if (notObfuscated()) {
			throw new UnsupportedOperationException("Cannot add remapper extension in a non-obfuscated environment");
		}

		final ObjectFactory objectFactory = getProject().getObjects();
		final RemapperExtensionHolder holder;

		if (parametersClass != RemapperParameters.None.class) {
			T parameters = objectFactory.newInstance(parametersClass);
			parameterAction.execute(parameters);
			holder = objectFactory.newInstance(RemapperExtensionHolder.class, parameters);
		} else {
			holder = objectFactory.newInstance(RemapperExtensionHolder.class, RemapperParameters.None.INSTANCE);
		}

		holder.getRemapperExtensionClass().set(remapperExtensionClass.getName());
		remapperExtensions.add(holder);
	}

	@Override
	public Provider<String> getMinecraftVersion() {
		return getProject().provider(() -> LoomGradleExtension.get(getProject()).getMinecraftProvider().minecraftVersion());
	}

	@Override
	public FileCollection getNamedMinecraftJars() {
		final ConfigurableFileCollection jars = getProject().getObjects().fileCollection();
		jars.from(getProject().provider(() -> LoomGradleExtension.get(getProject()).getMinecraftJars(MappingsNamespace.NAMED)));
		return jars;
	}

	private boolean notObfuscated() {
		return LoomGradleExtension.get(getProject()).disableObfuscation();
	}

	@Override
	public void silentMojangMappingsLicense() {
		try {
			this.silentMojangMappingsLicense.set(true);
		} catch (IllegalStateException e) {
			throw new IllegalStateException("loom.silentMojangMappingsLicense() must be called before its value is read, usually with loom.layered {}.", e);
		}
	}

	@Override
	public boolean isSilentMojangMappingsLicenseEnabled() {
		return silentMojangMappingsLicense.get();
	}

	@Override
	public Provider<ModPlatform> getPlatform() {
		return platform;
	}

	@Override
	public void setGenerateSrgTiny(Boolean generateSrgTiny) {
		if (isNeoForge()) {
			// This is unsupported because supporting the full 2x2 combination of
			//  [no extra NS] [SRG]
			//  [mojang]      [SRG+mojang]
			// is a bit verbose to support.
			throw new UnsupportedOperationException("SRG is not supported on NeoForge.");
		}

		this.generateSrgTiny = generateSrgTiny;
	}

	@Override
	public boolean shouldGenerateSrgTiny() {
		if (generateSrgTiny != null) {
			return generateSrgTiny;
		}

		return isForge();
	}

	@Override
	public List<String> getTasksBeforeRun() {
		return tasksBeforeRun;
	}

	@Override
	public List<Consumer<RunConfig>> getSettingsPostEdit() {
		return settingsPostEdit;
	}

	@Override
	public void forge(Action<ForgeExtensionAPI> action) {
		action.execute(getForge());
	}

	@Override
	public void neoForge(Action<NeoForgeExtensionAPI> action) {
		action.execute(getNeoForge());
	}

	// This is here to ensure that LoomGradleExtensionApiImpl compiles without any unimplemented methods
	private final class EnsureCompile extends LoomGradleExtensionApiImpl {
		private EnsureCompile() {
			super(null, null);
			throw new RuntimeException();
		}

		@Override
		public DeprecationHelper getDeprecationHelper() {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		protected Project getProject() {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		protected LoomFiles getFiles() {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		protected <T extends IntermediateMappingsProvider> void configureIntermediateMappingsProviderInternal(T provider) {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		public MixinExtension getMixin() {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		public ForgeExtensionAPI getForge() {
			throw new RuntimeException("Yeah... something is really wrong");
		}

		@Override
		public NeoForgeExtensionAPI getNeoForge() {
			throw new RuntimeException("Yeah... something is really wrong");
		}
	}
}
