/*
 * This file is part of architectury.
 * Copyright (C) 2020, 2021, 2022 architectury
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package dev.architectury.registry.level.biome.fabric;

import com.google.common.base.Predicates;
import com.google.common.collect.Lists;
import com.mojang.datafixers.util.Either;
import dev.architectury.hooks.level.biome.*;
import dev.architectury.registry.level.biome.BiomeModifications.BiomeContext;
import net.fabricmc.fabric.api.biome.v1.BiomeModification;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext.EffectsContext;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext.GenerationSettingsContext;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext.SpawnSettingsContext;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext.WeatherContext;
import net.fabricmc.fabric.api.biome.v1.BiomeSelectionContext;
import net.fabricmc.fabric.api.biome.v1.ModificationPhase;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1959;
import net.minecraft.class_1959.class_5484;
import net.minecraft.class_2893;
import net.minecraft.class_2922;
import net.minecraft.class_2960;
import net.minecraft.class_4763.class_5486;
import net.minecraft.class_5321;
import net.minecraft.class_5483;
import net.minecraft.class_6796;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.function.Predicate;

import static net.fabricmc.fabric.api.biome.v1.BiomeModificationContext.*;

public class BiomeModificationsImpl {
    private static final class_2960 FABRIC_MODIFICATION = class_2960.method_60655("architectury", "fabric_modification");
    private static final List<Pair<Predicate<BiomeContext>, BiConsumer<BiomeContext, BiomeProperties.Mutable>>> ADDITIONS = Lists.newArrayList();
    private static final List<Pair<Predicate<BiomeContext>, BiConsumer<BiomeContext, BiomeProperties.Mutable>>> POST_PROCESSING = Lists.newArrayList();
    private static final List<Pair<Predicate<BiomeContext>, BiConsumer<BiomeContext, BiomeProperties.Mutable>>> REMOVALS = Lists.newArrayList();
    private static final List<Pair<Predicate<BiomeContext>, BiConsumer<BiomeContext, BiomeProperties.Mutable>>> REPLACEMENTS = Lists.newArrayList();
    
    public static void addProperties(Predicate<BiomeContext> predicate, BiConsumer<BiomeContext, BiomeProperties.Mutable> modifier) {
        ADDITIONS.add(Pair.of(predicate, modifier));
    }
    
    public static void postProcessProperties(Predicate<BiomeContext> predicate, BiConsumer<BiomeContext, BiomeProperties.Mutable> modifier) {
        POST_PROCESSING.add(Pair.of(predicate, modifier));
    }
    
    public static void removeProperties(Predicate<BiomeContext> predicate, BiConsumer<BiomeContext, BiomeProperties.Mutable> modifier) {
        REMOVALS.add(Pair.of(predicate, modifier));
    }
    
    public static void replaceProperties(Predicate<BiomeContext> predicate, BiConsumer<BiomeContext, BiomeProperties.Mutable> modifier) {
        REPLACEMENTS.add(Pair.of(predicate, modifier));
    }
    
    static {
        var modification = net.fabricmc.fabric.api.biome.v1.BiomeModifications.create(FABRIC_MODIFICATION);
        registerModification(modification, ModificationPhase.ADDITIONS, ADDITIONS);
        registerModification(modification, ModificationPhase.POST_PROCESSING, POST_PROCESSING);
        registerModification(modification, ModificationPhase.REMOVALS, REMOVALS);
        registerModification(modification, ModificationPhase.REPLACEMENTS, REPLACEMENTS);
    }
    
    private static void registerModification(BiomeModification modification, ModificationPhase phase, List<Pair<Predicate<BiomeContext>, BiConsumer<BiomeContext, BiomeProperties.Mutable>>> list) {
        modification.add(phase, Predicates.alwaysTrue(), (biomeSelectionContext, biomeModificationContext) -> {
            var biomeContext = wrapSelectionContext(biomeSelectionContext);
            var mutableBiome = wrapMutableBiome(biomeSelectionContext.getBiome(), biomeModificationContext);
            for (var pair : list) {
                if (pair.getLeft().test(biomeContext)) {
                    pair.getRight().accept(biomeContext, mutableBiome);
                }
            }
        });
    }
    
    private static BiomeContext wrapSelectionContext(BiomeSelectionContext context) {
        return new BiomeContext() {
            BiomeProperties properties = BiomeHooks.getBiomeProperties(context.getBiome());
            
            @Override
            public Optional<class_2960> getKey() {
                return Optional.ofNullable(context.getBiomeKey().method_29177());
            }
            
            @Override
            public BiomeProperties getProperties() {
                return properties;
            }
            
            @Override
            public boolean hasTag(class_6862<class_1959> tag) {
                return context.hasTag(tag);
            }
        };
    }
    
    private static BiomeProperties.Mutable wrapMutableBiome(class_1959 biome, BiomeModificationContext context) {
        return new BiomeHooks.MutableBiomeWrapped(
                biome,
                wrapWeather(biome, context.getWeather()),
                wrapEffects(biome, context.getEffects()),
                new MutableGenerationProperties(biome, context.getGenerationSettings()),
                new MutableSpawnProperties(biome, context.getSpawnSettings())
        ) {
        };
    }
    
    private static class MutableGenerationProperties extends BiomeHooks.GenerationSettingsWrapped implements GenerationProperties.Mutable {
        protected final GenerationSettingsContext context;
        
        public MutableGenerationProperties(class_1959 biome, GenerationSettingsContext context) {
            super(biome);
            this.context = context;
        }
        
        @Override
        public Mutable addFeature(class_2893.class_2895 decoration, class_6880<class_6796> feature) {
            Either<class_5321<class_6796>, class_6796> unwrap = feature.method_40229();
            if (unwrap.left().isPresent()) {
                this.context.addFeature(decoration, unwrap.left().get());
            } else {
                throw new UnsupportedOperationException("Cannot add a feature that is not registered: " + unwrap.right().orElseThrow());
            }
            return this;
        }
        
        @Override
        public Mutable addFeature(class_2893.class_2895 decoration, class_5321<class_6796> feature) {
            this.context.addFeature(decoration, feature);
            return this;
        }
        
        @Override
        public Mutable addCarver(class_6880<class_2922<?>> feature) {
            Either<class_5321<class_2922<?>>, class_2922<?>> unwrap = feature.method_40229();
            if (unwrap.left().isPresent()) {
                this.context.addCarver(unwrap.left().get());
            } else {
                throw new UnsupportedOperationException("Cannot add a carver that is not registered: " + unwrap.right().orElseThrow());
            }
            return this;
        }
        
        @Override
        public Mutable addCarver(class_5321<class_2922<?>> feature) {
            this.context.addCarver(feature);
            return this;
        }
        
        @Override
        public Mutable removeFeature(class_2893.class_2895 decoration, class_5321<class_6796> feature) {
            context.removeFeature(decoration, feature);
            return this;
        }
        
        @Override
        public Mutable removeCarver(class_5321<class_2922<?>> feature) {
            context.removeCarver(feature);
            return this;
        }
    }
    
    private static class MutableSpawnProperties extends BiomeHooks.SpawnSettingsWrapped implements SpawnProperties.Mutable {
        protected final SpawnSettingsContext context;
        
        public MutableSpawnProperties(class_1959 biome, SpawnSettingsContext context) {
            super(biome);
            this.context = context;
        }
        
        @Override
        public Mutable setCreatureProbability(float probability) {
            context.setCreatureSpawnProbability(probability);
            return this;
        }
        
        @Override
        public Mutable addSpawn(class_1311 category, class_5483.class_1964 data, int weight) {
            context.addSpawn(category, data, weight);
            return this;
        }
        
        @Override
        public boolean removeSpawns(BiPredicate<class_1311, class_5483.class_1964> predicate) {
            return context.removeSpawns(predicate);
        }
        
        @Override
        public Mutable setSpawnCost(class_1299<?> entityType, class_5483.class_5265 cost) {
            context.setSpawnCost(entityType, cost.comp_1308(), cost.comp_1307());
            return this;
        }
        
        @Override
        public Mutable setSpawnCost(class_1299<?> entityType, double charge, double energyBudget) {
            context.setSpawnCost(entityType, charge, energyBudget);
            return this;
        }
        
        @Override
        public Mutable clearSpawnCost(class_1299<?> entityType) {
            context.clearSpawnCost(entityType);
            return this;
        }
    }
    
    private static ClimateProperties.Mutable wrapWeather(class_1959 biome, WeatherContext context) {
        return new BiomeHooks.ClimateWrapped(biome) {
            @Override
            public ClimateProperties.Mutable setHasPrecipitation(boolean hasPrecipitation) {
                context.setPrecipitation(hasPrecipitation);
                return this;
            }
            
            @Override
            public ClimateProperties.Mutable setTemperature(float temperature) {
                context.setTemperature(temperature);
                return this;
            }
            
            @Override
            public ClimateProperties.Mutable setTemperatureModifier(class_5484 temperatureModifier) {
                context.setTemperatureModifier(temperatureModifier);
                return this;
            }
            
            @Override
            public ClimateProperties.Mutable setDownfall(float downfall) {
                context.setDownfall(downfall);
                return this;
            }
        };
    }
    
    private static EffectsProperties.Mutable wrapEffects(class_1959 biome, EffectsContext context) {
        return new BiomeHooks.EffectsWrapped(biome) {
            @Override
            public EffectsProperties.Mutable setWaterColor(int color) {
                context.setWaterColor(color);
                return this;
            }
            
            @Override
            public EffectsProperties.Mutable setFoliageColorOverride(@Nullable Integer colorOverride) {
                context.setFoliageColor(Optional.ofNullable(colorOverride));
                return this;
            }
            
            @Override
            public EffectsProperties.Mutable setDryFoliageColorOverride(@Nullable Integer colorOverride) {
                context.setDryFoliageColor(Optional.ofNullable(colorOverride));
                return this;
            }
            
            @Override
            public EffectsProperties.Mutable setGrassColorOverride(@Nullable Integer colorOverride) {
                context.setGrassColor(Optional.ofNullable(colorOverride));
                return this;
            }
            
            @Override
            public EffectsProperties.Mutable setGrassColorModifier(class_5486 modifier) {
                context.setGrassColorModifier(modifier);
                return this;
            }
        };
    }
    
}

