/*
 * This file is part of architectury.
 * Copyright (C) 2020, 2021, 2022 architectury
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package dev.architectury.fluid;

import com.mojang.serialization.Codec;
import dev.architectury.hooks.fluid.FluidStackHooks;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_6880;
import net.minecraft.class_7225;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.class_9322;
import net.minecraft.class_9323;
import net.minecraft.class_9326;
import net.minecraft.class_9331;
import net.minecraft.class_9335;
import net.minecraft.core.component.*;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

public final class FluidStack implements class_9322 {
    private static final FluidStackAdapter<Object> ADAPTER = adapt(FluidStack::getValue, FluidStack::new);
    private static final FluidStack EMPTY = new FluidStack(() -> class_3612.field_15906, 0, class_9326.field_49588);
    public static final Codec<FluidStack> CODEC = ADAPTER.codec();
    public static final class_9139<class_9129, FluidStack> STREAM_CODEC = ADAPTER.streamCodec();
    
    private final Object value;
    
    private FluidStack(Supplier<class_3611> fluid, long amount, class_9326 patch) {
        this(ADAPTER.create(fluid, amount, patch));
    }
    
    private FluidStack(Object value) {
        this.value = Objects.requireNonNull(value);
    }
    
    private Object getValue() {
        return value;
    }
    
    @ExpectPlatform
    private static FluidStackAdapter<Object> adapt(Function<FluidStack, Object> toValue, Function<Object, FluidStack> fromValue) {
        throw new AssertionError();
    }
    
    @ApiStatus.Internal
    public interface FluidStackAdapter<T> {
        T create(Supplier<class_3611> fluid, long amount, @Nullable class_9326 patch);
        
        Supplier<class_3611> getRawFluidSupplier(T object);
        
        class_3611 getFluid(T object);
        
        long getAmount(T object);
        
        void setAmount(T object, long amount);
        
        class_9326 getPatch(T value);
        
        class_9335 getComponents(T value);
        
        void applyComponents(T value, class_9326 patch);
        
        void applyComponents(T value, class_9323 patch);
        
        @Nullable <D> D set(T value, class_9331<D> type, @Nullable D component);
        
        @Nullable <D> D remove(T value, class_9331<? extends D> type);
        
        @Nullable <D> D update(T value, class_9331<D> type, D component, UnaryOperator<D> updater);
        
        @Nullable <D, U> D update(T value, class_9331<D> type, D component, U updateContext, BiFunction<D, U, D> updater);
        
        T copy(T value);
        
        int hashCode(T value);
        
        Codec<FluidStack> codec();
        
        class_9139<class_9129, FluidStack> streamCodec();
    }
    
    public static FluidStack empty() {
        return EMPTY;
    }
    
    public static FluidStack create(class_3611 fluid, long amount, class_9326 patch) {
        if (fluid == class_3612.field_15906 || amount <= 0) return empty();
        return create(() -> fluid, amount, patch);
    }
    
    public static FluidStack create(class_3611 fluid, long amount) {
        return create(fluid, amount, class_9326.field_49588);
    }
    
    public static FluidStack create(Supplier<class_3611> fluid, long amount, class_9326 patch) {
        if (amount <= 0) return empty();
        return new FluidStack(fluid, amount, patch);
    }
    
    public static FluidStack create(Supplier<class_3611> fluid, long amount) {
        return create(fluid, amount, class_9326.field_49588);
    }
    
    public static FluidStack create(class_6880<class_3611> fluid, long amount, class_9326 patch) {
        return create(fluid.comp_349(), amount, patch);
    }
    
    public static FluidStack create(class_6880<class_3611> fluid, long amount) {
        return create(fluid.comp_349(), amount, class_9326.field_49588);
    }
    
    public static FluidStack create(FluidStack stack, long amount) {
        return create(stack.getRawFluidSupplier(), amount, stack.getPatch());
    }
    
    public static long bucketAmount() {
        return FluidStackHooks.bucketAmount();
    }
    
    public class_3611 getFluid() {
        return isEmpty() ? class_3612.field_15906 : getRawFluid();
    }
    
    @Nullable
    public class_3611 getRawFluid() {
        return ADAPTER.getFluid(value);
    }
    
    public Supplier<class_3611> getRawFluidSupplier() {
        return ADAPTER.getRawFluidSupplier(value);
    }
    
    public boolean isEmpty() {
        return getRawFluid() == class_3612.field_15906 || ADAPTER.getAmount(value) <= 0;
    }
    
    public long getAmount() {
        return isEmpty() ? 0 : ADAPTER.getAmount(value);
    }
    
    public void setAmount(long amount) {
        ADAPTER.setAmount(value, amount);
    }
    
    public void grow(long amount) {
        setAmount(getAmount() + amount);
    }
    
    public void shrink(long amount) {
        setAmount(getAmount() - amount);
    }
    
    public class_9326 getPatch() {
        return ADAPTER.getPatch(value);
    }
    
    @Override
    public class_9335 method_57353() {
        return ADAPTER.getComponents(value);
    }
    
    public void applyComponents(class_9326 patch) {
        ADAPTER.applyComponents(value, patch);
    }
    
    public void applyComponents(class_9323 patch) {
        ADAPTER.applyComponents(value, patch);
    }
    
    @Nullable
    public <T> T set(class_9331<T> type, @Nullable T component) {
        return ADAPTER.set(value, type, component);
    }
    
    @Nullable
    public <T> T remove(class_9331<? extends T> type) {
        return ADAPTER.remove(value, type);
    }
    
    @Nullable
    public <T> T update(class_9331<T> type, T component, UnaryOperator<T> updater) {
        return ADAPTER.update(value, type, component, updater);
    }
    
    @Nullable
    public <T, U> T update(class_9331<T> type, T component, U updateContext, BiFunction<T, U, T> updater) {
        return ADAPTER.update(value, type, component, updateContext, updater);
    }
    
    public class_2561 getName() {
        return FluidStackHooks.getName(this);
    }
    
    public String getTranslationKey() {
        return FluidStackHooks.getTranslationKey(this);
    }
    
    public FluidStack copy() {
        return new FluidStack(ADAPTER.copy(value));
    }
    
    @Override
    public int hashCode() {
        return ADAPTER.hashCode(value);
    }
    
    @Override
    public boolean equals(Object o) {
        if (!(o instanceof FluidStack)) {
            return false;
        }
        return isFluidStackEqual((FluidStack) o);
    }
    
    public boolean isFluidStackEqual(FluidStack other) {
        return getFluid() == other.getFluid() && getAmount() == other.getAmount() && isComponentEqual(other);
    }
    
    public boolean isFluidEqual(FluidStack other) {
        return getFluid() == other.getFluid();
    }
    
    public boolean isComponentEqual(FluidStack other) {
        var patch = getPatch();
        var otherPatch = other.getPatch();
        return Objects.equals(patch, otherPatch);
    }
    
    public static FluidStack read(class_9129 buf) {
        return FluidStackHooks.read(buf);
    }
    
    public static Optional<FluidStack> read(class_7225.class_7874 provider, class_2520 tag) {
        return FluidStackHooks.read(provider, tag);
    }
    
    public void write(class_9129 buf) {
        FluidStackHooks.write(this, buf);
    }
    
    public class_2520 write(class_7225.class_7874 provider, class_2520 tag) {
        return FluidStackHooks.write(provider, this, tag);
    }
    
    public FluidStack copyWithAmount(long amount) {
        if (isEmpty()) return this;
        return new FluidStack(getRawFluidSupplier(), amount, getPatch());
    }
    
    @ApiStatus.Internal
    public static void init() {
        // classloading my beloved 😍
        // please don't use this by the way
    }
}
